package deploystage

import (
	"context"
	"net/http"

	corev1 "k8s.io/api/core/v1"
	"sigs.k8s.io/controller-runtime/pkg/client"
	logf "sigs.k8s.io/controller-runtime/pkg/log"
	"sigs.k8s.io/controller-runtime/pkg/webhook/admission"

	"a.yandex-team.ru/infra/infractl/clients/abc"
	dv1 "a.yandex-team.ru/infra/infractl/controllers/deploy/api/stage/v1"
	dclient "a.yandex-team.ru/infra/infractl/internal/deploy/client"
	"a.yandex-team.ru/infra/infractl/internal/secrets"
	"a.yandex-team.ru/infra/infractl/webhooks/internal/objects"
	hooksecrets "a.yandex-team.ru/infra/infractl/webhooks/internal/secrets"
	"a.yandex-team.ru/library/go/yandex/tvm/tvmauth"
	"a.yandex-team.ru/library/go/yandex/yav/httpyav"
)

var whlog = logf.Log.WithName("ds-webhook")

// Validator validates DeployStages
type Validator struct {
	Client       client.Client
	DeployClient *dclient.DeployClient
	decoder      *admission.Decoder
	AbcClient    *abc.Client
	TvmClient    *tvmauth.Client
	YavClient    *httpyav.Client
	YpCluster    string
	TvmID        uint64
	DevMode      hooksecrets.DevMode
}

func (v *Validator) GetYpToken(ctx context.Context, ns *corev1.Namespace) (string, error) {
	if v.DevMode.DevMode {
		return v.DevMode.YpToken, nil
	} else {
		return secrets.LoadProviderToken(v, ctx, ns, "yp", v.GetTvmID())
	}
}

func (v *Validator) GetTvmClient() *tvmauth.Client {
	return v.TvmClient
}

func (v *Validator) GetYavClient() *httpyav.Client {
	return v.YavClient
}

func (v *Validator) GetK8sClient() client.Client {
	return v.Client
}

func (v *Validator) Handle(ctx context.Context, req admission.Request) admission.Response {
	kStage := &dv1.DeployStage{}

	err := v.decoder.Decode(req, kStage)
	if err != nil {
		return admission.Errored(http.StatusBadRequest, err)
	}

	log := whlog.WithValues("namespace", kStage.Namespace, "name", kStage.Name)

	log.V(1).Info("validating DeployStage")

	current, rsp := objects.LoadCurrent(ctx, log, v.Client, kStage)
	if !rsp.Allowed {
		return rsp
	}
	var curStage *dv1.DeployStage
	if current != nil {
		curStage = current.(*dv1.DeployStage)
	}

	if rsp = v.validateStage(ctx, log, curStage, kStage); !rsp.Allowed {
		return rsp
	}

	if rsp = v.validateSecrets(ctx, log, curStage, kStage); !rsp.Allowed {
		return rsp
	}

	return admission.Allowed("")
}

func (v *Validator) InjectDecoder(d *admission.Decoder) error {
	v.decoder = d
	return nil
}
