package secrets

import (
	"context"
	"fmt"
	"net/http"
	"strings"

	"github.com/go-logr/logr"
	corev1 "k8s.io/api/core/v1"
	"k8s.io/apimachinery/pkg/types"
	"k8s.io/apimachinery/pkg/util/sets"
	"sigs.k8s.io/controller-runtime/pkg/client"
	"sigs.k8s.io/controller-runtime/pkg/webhook/admission"

	"a.yandex-team.ru/infra/infractl/clients/blackbox"
	"a.yandex-team.ru/infra/infractl/internal/secrets"
)

type DevMode struct {
	DevMode bool
	YpToken string
}

type validator interface {
	CollectSecrets(obj client.Object) sets.String
	GetTvmID() uint64
}

type webhookValidator interface {
	secrets.SecretReader
	validator
	GetYpToken(ctx context.Context, ns *corev1.Namespace) (string, error)
}

func failure(err error) admission.Response {
	return admission.Errored(http.StatusBadRequest, err)
}

func ValidateSecrets(
	ctx context.Context,
	log logr.Logger,
	v webhookValidator,
	curObject, newObject client.Object,
) admission.Response {
	currentSecrets := v.CollectSecrets(curObject)
	newSecrets := v.CollectSecrets(newObject)
	missingSecrets := newSecrets.Difference(currentSecrets)
	var notDelegatedSecrets []string

	var robotUID uint64
	if len(missingSecrets) > 0 {
		ns := &corev1.Namespace{}
		if err := client.IgnoreNotFound(v.GetK8sClient().Get(ctx, types.NamespacedName{Name: newObject.GetNamespace()}, ns)); err != nil {
			log.Error(err, "failed to get namespace")
			return failure(fmt.Errorf("failed to get namespace %s: %w", newObject.GetNamespace(), err))
		}
		token, err := v.GetYpToken(ctx, ns)
		if err != nil {
			return failure(fmt.Errorf("failed to get YP token from namespace: %v", err))
		}

		bbClient, err := blackbox.MakeClient(v.GetTvmClient())
		if err != nil {
			return failure(fmt.Errorf("failed to create blackbox client: %v", err))
		}
		_, robotUID, err = bbClient.GetLoginFromOauth(ctx, token, GetIP())
		if err != nil {
			return failure(fmt.Errorf("failed to get YP user login: %v", err))
		}
	}

	for s := range missingSecrets {
		delegated, err := secrets.CheckSecretReadable(ctx, v.GetYavClient(), robotUID, s)
		if err != nil {
			return failure(fmt.Errorf("failed to check secret %q: %w", s, err))
		}
		if !delegated {
			notDelegatedSecrets = append(notDelegatedSecrets, s)
		}
	}

	if len(notDelegatedSecrets) > 0 {
		return admission.Errored(
			http.StatusFailedDependency,
			fmt.Errorf("secrets not delegated: %v", strings.Join(notDelegatedSecrets, ", ")),
		)
	}

	return admission.Allowed("")

}
