package main

import (
	"flag"
	"io/ioutil"
	"os"

	"gopkg.in/yaml.v2"
	"k8s.io/client-go/kubernetes/scheme"
	_ "k8s.io/client-go/plugin/pkg/client/auth/gcp"
	ctrl "sigs.k8s.io/controller-runtime"
	"sigs.k8s.io/controller-runtime/pkg/client/config"
	"sigs.k8s.io/controller-runtime/pkg/log"
	"sigs.k8s.io/controller-runtime/pkg/log/zap"
	"sigs.k8s.io/controller-runtime/pkg/manager"
	"sigs.k8s.io/controller-runtime/pkg/manager/signals"
	"sigs.k8s.io/controller-runtime/pkg/webhook"

	"a.yandex-team.ru/infra/infractl/clients/abc"
	dclient "a.yandex-team.ru/infra/infractl/internal/deploy/client"
	_ "a.yandex-team.ru/infra/infractl/internal/scheme"
	"a.yandex-team.ru/infra/infractl/webhooks/deployproject"
	"a.yandex-team.ru/infra/infractl/webhooks/deploystage"
	"a.yandex-team.ru/infra/infractl/webhooks/internal/secrets"
	"a.yandex-team.ru/infra/infractl/webhooks/namespace"
	"a.yandex-team.ru/infra/infractl/webhooks/runtime"
	"a.yandex-team.ru/library/go/core/log/nop"
	"a.yandex-team.ru/library/go/yandex/blackbox/httpbb"
	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/library/go/yandex/tvm/tvmauth"
	"a.yandex-team.ru/library/go/yandex/yav/httpyav"
	"a.yandex-team.ru/yp/go/yp"
)

func init() {
	log.SetLogger(zap.New())
}

type abcConfig struct {
	URL string `yaml:"api_url"`
}

type ypConfig struct {
	Cluster string `yaml:"cluster"`
}

type serverConfig struct {
	Port uint16 `yaml:"port"`
}

type tvmConfig struct {
	YavTvmID  uint64 `yaml:"yav_tvm_id"`
	TvmID     uint64 `yaml:"tvm_id"`
	TvmSecret string `yaml:"tvm_secret"`
}

type webhookConfig struct {
	CertDir string       `yaml:"cert_dir"`
	ABC     abcConfig    `yaml:"abc"`
	YP      ypConfig     `yaml:"yp"`
	TVM     tvmConfig    `yaml:"tvm"`
	Server  serverConfig `yaml:"server"`
}

const (
	envTokenName = "YP_TOKEN"
)

func main() {
	devMode := flag.Bool("dev-mode", false,
		"If specified then developers mode will be used. "+
			"OAuth token will be taken from "+envTokenName+" environment variable.")
	configFile := flag.String("config", "",
		"The webhook will load its initial configuration from this file. "+
			"Omit this flag to use the default configuration values. "+
			"Command-line flags override configuration from this file.")

	flag.Parse()

	entryLog := log.Log.WithName("entrypoint")
	var YpToken = ""
	if *devMode {
		YpToken = os.Getenv(envTokenName)
		if YpToken == "" {
			entryLog.Info("unable to get env variable", "envName", envTokenName)
			os.Exit(1)
		}
	}
	// Setup a Manager
	entryLog.Info("setting up manager")

	// Read webhook config
	data, err := ioutil.ReadFile(*configFile)
	if err != nil {
		entryLog.Error(err, "unable to read webhook config file")
		os.Exit(1)
	}
	cfg := &webhookConfig{}
	err = yaml.Unmarshal(data, cfg)
	if err != nil {
		entryLog.Error(err, "unable to parse webhook config file")
		os.Exit(1)
	}
	cfg.TVM.TvmSecret = os.Getenv("TVM_SECRET")
	if cfg.TVM.TvmSecret == "" {
		entryLog.Error(err, "unable to get TVM_SECRET env variable")
		os.Exit(1)
	}

	options := ctrl.Options{
		Scheme:  scheme.Scheme,
		CertDir: cfg.CertDir,
		Port:    int(cfg.Server.Port),
	}
	kcfg := config.GetConfigOrDie()
	kcfg.BearerToken = os.Getenv("KUBE_OAUTH_TOKEN")
	mgr, err := manager.New(kcfg, options)
	if err != nil {
		entryLog.Error(err, "unable to set up overall controller manager")
		os.Exit(1)
	}

	ypClient, err := yp.NewClient(
		cfg.YP.Cluster,
		yp.WithAuthToken(os.Getenv("YP_OAUTH_TOKEN")),
		// orm.WithLogger(log), // TODO(torkve) need to adapt logr to arcadian logger somehow
	)
	if err != nil {
		entryLog.Error(err, "unable to set up YP client")
		os.Exit(1)
	}
	// We need only ypClient because we use only Fetch method
	deployClient := dclient.NewDeployClient(ypClient, nil, nil)

	tvmSettings := tvmauth.TvmAPISettings{
		SelfID: tvm.ClientID(cfg.TVM.TvmID),
		ServiceTicketOptions: tvmauth.NewAliasesOptions(
			cfg.TVM.TvmSecret,
			map[string]tvm.ClientID{
				"yav":      tvm.ClientID(cfg.TVM.YavTvmID),
				"blackbox": httpbb.IntranetEnvironment.TvmID,
			},
		),
	}
	tvmc, err := tvmauth.NewAPIClient(tvmSettings, &nop.Logger{})
	if err != nil {
		entryLog.Error(err, "unable to set up TVM client")
		os.Exit(1)
	}
	yavc, err := httpyav.NewClient()
	if err != nil {
		entryLog.Error(err, "unable to set up yav client")
		os.Exit(1)
	}

	// Setup webhooks
	abcToken := os.Getenv("ABC_OAUTH_TOKEN")
	if abcToken == "" {
		entryLog.Error(err, "unable to get ABC_OAUTH_TOKEN env variable")
		os.Exit(1)
	}
	abcClient := abc.NewClient(abcToken)

	entryLog.Info("setting up webhook server")
	hookServer := mgr.GetWebhookServer()

	entryLog.Info("registering webhooks to the webhook server")
	hookServer.Register("/validate-v1-namespace", &webhook.Admission{Handler: &namespace.NamespaceValidator{Client: mgr.GetClient(), ABCClient: abcClient}})

	dsValidator := &deploystage.Validator{
		Client:       mgr.GetClient(),
		AbcClient:    abcClient,
		DeployClient: deployClient,
		TvmClient:    tvmc,
		YavClient:    yavc,
		YpCluster:    cfg.YP.Cluster,
		TvmID:        cfg.TVM.TvmID,
		DevMode:      secrets.DevMode{DevMode: *devMode, YpToken: YpToken},
	}
	hookServer.Register(
		"/validate-v1-deploystage",
		&webhook.Admission{Handler: dsValidator},
	)

	runtimeValidator := &runtime.Validator{
		Client:    mgr.GetClient(),
		TvmClient: tvmc,
		YavClient: yavc,
		TvmID:     cfg.TVM.TvmID,
		DevMode:   secrets.DevMode{DevMode: *devMode, YpToken: YpToken},
	}
	hookServer.Register(
		"/validate-v1-runtime",
		&webhook.Admission{Handler: runtimeValidator},
	)

	hookServer.Register(
		"/validate-v1-deployproject",
		&webhook.Admission{Handler: deployproject.NewDeployProjectValidator(mgr.GetClient(), deployClient, abcClient)},
	)

	entryLog.Info("starting manager")
	if err := mgr.Start(signals.SetupSignalHandler()); err != nil {
		entryLog.Error(err, "unable to run manager")
		os.Exit(1)
	}
}
