package namespace

import (
	"context"
	"net/http"

	corev1 "k8s.io/api/core/v1"
	"sigs.k8s.io/controller-runtime/pkg/client"
	logf "sigs.k8s.io/controller-runtime/pkg/log"
	"sigs.k8s.io/controller-runtime/pkg/webhook/admission"

	"a.yandex-team.ru/infra/infractl/clients/abc"
	"a.yandex-team.ru/infra/infractl/internal/validation"
)

var whlog = logf.Log.WithName("ns-webhook")

// NamespaceValidator validates Namespaces
type NamespaceValidator struct {
	Client    client.Client
	ABCClient *abc.Client
	decoder   *admission.Decoder
}

// Handle admits a namespace if a specific annotation exists.
func (v *NamespaceValidator) Handle(ctx context.Context, req admission.Request) admission.Response {
	ns := &corev1.Namespace{}

	err := v.decoder.Decode(req, ns)
	if err != nil {
		return admission.Errored(http.StatusBadRequest, err)
	}

	whlog.V(1).Info("validating namespace", "namespace", ns.Name)

	isValid, msg, err := validation.ValidateNamespace(v.ABCClient, whlog, ns, req.UserInfo.Username)
	if err != nil {
		return admission.Errored(http.StatusBadRequest, err)
	}
	if !isValid {
		return admission.Denied(msg)
	}
	return admission.Allowed("")
}

// NamespaceValidator implements admission.DecoderInjector.
// A decoder will be automatically injected.

// InjectDecoder injects the decoder.
func (v *NamespaceValidator) InjectDecoder(d *admission.Decoder) error {
	v.decoder = d
	return nil
}
