package runtime

import (
	"context"
	"net/http"

	corev1 "k8s.io/api/core/v1"
	"sigs.k8s.io/controller-runtime/pkg/client"
	logf "sigs.k8s.io/controller-runtime/pkg/log"
	"sigs.k8s.io/controller-runtime/pkg/webhook/admission"

	rv1 "a.yandex-team.ru/infra/infractl/controllers/runtime/api/v1"
	"a.yandex-team.ru/infra/infractl/internal/secrets"
	"a.yandex-team.ru/infra/infractl/webhooks/internal/objects"
	hooksecrets "a.yandex-team.ru/infra/infractl/webhooks/internal/secrets"
	"a.yandex-team.ru/library/go/yandex/tvm/tvmauth"
	"a.yandex-team.ru/library/go/yandex/yav/httpyav"
)

var whlog = logf.Log.WithName("ds-webhook")

// Validator validates Runtimes
type Validator struct {
	Client    client.Client
	decoder   *admission.Decoder
	TvmClient *tvmauth.Client
	YavClient *httpyav.Client
	TvmID     uint64
	DevMode   hooksecrets.DevMode
}

func (v *Validator) GetYpToken(ctx context.Context, ns *corev1.Namespace) (string, error) {
	if v.DevMode.DevMode {
		return v.DevMode.YpToken, nil
	} else {
		return secrets.LoadProviderToken(v, ctx, ns, "yp", v.GetTvmID())
	}
}

func (v *Validator) GetTvmClient() *tvmauth.Client {
	return v.TvmClient
}

func (v *Validator) GetYavClient() *httpyav.Client {
	return v.YavClient
}

func (v *Validator) GetK8sClient() client.Client {
	return v.Client
}

func (v *Validator) InjectDecoder(d *admission.Decoder) error {
	v.decoder = d
	return nil
}

func (v *Validator) Handle(ctx context.Context, req admission.Request) admission.Response {
	kRuntime := &rv1.Runtime{}
	err := v.decoder.Decode(req, kRuntime)
	if err != nil {
		return admission.Errored(http.StatusBadRequest, err)
	}

	log := whlog.WithValues("namespace", kRuntime.Namespace, "name", kRuntime.Name)

	log.V(1).Info("validating Runtime")

	current, rsp := objects.LoadCurrent(ctx, log, v.Client, kRuntime)
	if !rsp.Allowed {
		return rsp
	}
	var curRuntime *rv1.Runtime
	if current != nil {
		curRuntime = current.(*rv1.Runtime)
	}

	if rsp = v.validateSecrets(ctx, log, curRuntime, kRuntime); !rsp.Allowed {
		return rsp
	}

	return admission.Allowed("")
}
