#!/usr/bin/env python2.7
import json
import logging
import os
import re
import subprocess
import urllib2


log = logging.getLogger(__name__)
DEBUG_OUT = False


def run_process(cmds, fail=True, env=None, shell=False):
    """
    get command output
    """
    def out_debug(msg):
        if DEBUG_OUT:
            logging.debug('run_process: ' + msg)

    if isinstance(cmds, basestring):
        cmd_str = cmds
    else:
        cmd_str = ' '.join(cmds)

    # log.debug(DELIMITER)
    # log.debug('RUN: %s' % (cmd_str, ))

    if env:
        out_debug('env: %r' % (env, ))

    if env is None:  # not "if env:" because there can be empty dict: env={}
        local_env = dict(os.environ)
    else:
        local_env = dict(env)

    local_env.update({'LANG': 'en_US.UTF-8'})

    p = subprocess.Popen(cmds, close_fds=True, stdout=subprocess.PIPE, stderr=subprocess.STDOUT,
                         env=local_env, shell=shell)
    s = p.stdout.read()
    ret = p.wait()

    if fail and ret != 0:
        raise Exception("Process ret=%d: %s" % (ret, s))

    # log.debug('%s\n%s\n%s\n' % (DELIMITER_LITE, s, DELIMITER_LITE))
    return s


def split_by_empty_lines(lines):
    result = []
    local = list(lines)
    while True:
        current_record = []
        while local and local[0] == '':
            local.pop(0)
        while local and local[0] != '':
            current_record.append(local.pop(0))
        if current_record:
            result.append(current_record)
        if not local:
            break
    return result

INSTANCE_STATE_RE = re.compile(r'^(?P<service>\d+)\s+@\s+'
                               r'(?P<config_name>\S+)\s+'
                               r'current:\[(?P<current>[^\]]+)\]\s+'
                               r'target:\[(?P<target>[^\]]+)\]')
INSTANCE_CWD_RE = re.compile(r'^cwd:\[(?P<cwd>\S+)\]')
INSTANCE_MSG_RE = re.compile(r'message\s+:\s+(?P<message>.*?)\s*$')
INSTANCE_CWD_PARTS_RE = re.compile(r'^(?P<port>\d+)@(?P<host>\S+?(?:yandex\.ru|\.search\.yandex\.net))-'
                                   r'(?P<chksum>[a-f\d]{32})')


def parse_instance(instance_lines):
    """
    18644 @ testws-web_base-106#22909a933e5d723817d822d5e3580af5 current:[PREPARED] target:[PREPARED]
    cwd:[18644@ams1-1160.search.yandex.net-a1280408aa4aae92c32f31d79b762b76]
    message  :  sync successful
    """
    assert len(instance_lines) >= 2, 'wait for at least 3lines: %r' % (instance_lines, )
    data = {'_src': list(instance_lines)}

    match = INSTANCE_STATE_RE.match(instance_lines[0])
    if match:
        data['port'] = data['service'] = match.group('service')
        data['config_name'] = match.group('config_name')
        data['current'] = match.group('current')
        data['target'] = match.group('target')
    else:
        print 'cant parse line 0: [%r]' % (instance_lines, )
        return None

    match = INSTANCE_CWD_RE.match(instance_lines[1])
    if match:
        data['cwd'] = match.group('cwd')
    else:
        print 'cant parse line 1: [%r]' % (instance_lines, )
        return None

    data['message'] = ''

    if len(instance_lines) >= 3:
        match = INSTANCE_MSG_RE.match(instance_lines[2])
        if match:
            data['message'] = match.group('message')

    # try to parse hostname
    match = INSTANCE_CWD_PARTS_RE.match(data['cwd'])
    if match:
        data['host'] = match.group('host')
        data['instance.chksum'] = match.group('chksum')
        data['basehost'] = data['host'].split('.')[0]
    else:
        print 'cant extract hostname from [%s]' % (data['cwd'], )
        return None

    return data


def get_instances(raw_data):
    lines = [i.strip() for i in raw_data.splitlines()]
    lines = [i for i in lines if i != 'Done' and not i.startswith('#')]

    instances = split_by_empty_lines(lines)

    return [parse_instance(i_lines) for i_lines in instances]


def join_url_parts(a, b):
    return a.rstrip('/') + '/' + b.lstrip('/')


def load_json_from_url(relative_url):
    url = join_url_parts('http://localhost:25536/', relative_url)
    print url
    return json.loads(urllib2.urlopen(url).read())


def load_json_from_file(fn):
    with open(fn, 'rb') as fh:
        data = fh.read()
    return json.loads(data)


class IssAgentChecks(object):
    ACTIVE = 'ACTIVE'
    PREPARED = 'PREPARED'
    CHKSUM_IN_PROGRESS = 'Checksum is in progress'
    UNKNOWN = 'UNKNOWN'

    INSTANCES_DIR = '/db/iss3/instances'
    DUMP_JSON = 'dump.json'

    def __init__(self):
        # self.instances_json = load_json_from_url('/instances')
        # print 'instances_json:', len(self.instances_json)

        print '# terminal: call instances()'
        self.instances_raw = run_process("/db/iss3/cli.py instances", shell=True)
        self.instances = get_instances(self.instances_raw)
        self.dump_json_map = {}

    def load_dump_json_files(self):
        self.dump_json_map = {}

        for fn in os.listdir(self.INSTANCES_DIR):
            path = os.path.join(self.INSTANCES_DIR, fn)

            if not os.path.isdir(path):
                print 'not a dir:', path
                continue

            dump_json = os.path.join(path, self.DUMP_JSON)

            if not os.path.isfile(dump_json):
                print 'not found:', dump_json
                continue

            try:
                state = load_json_from_file(dump_json)
            except:
                print 'cant load JSON:', dump_json
                continue

            self.dump_json_map[fn] = state

    def do_check(self):
        self.check_running_agent()
        self.check_iss996()

    @staticmethod
    def print_result(lines):
        for i in lines:
            print i
        print

    def check_instance_iss996(self, instance):
        result = []

        try:
            current = instance['current']
            if current in [self.ACTIVE, self.PREPARED, self.CHKSUM_IN_PROGRESS, self.UNKNOWN]:
                cwd = instance['cwd']
                if not os.path.isdir(os.path.join(self.INSTANCES_DIR, cwd)):
                    result.append('instance dir not found: %s' % (instance['cwd']))
                self.iss996_visited_dirs.add(cwd)

            prefix = '%(port)s@%(basehost)s: ' % instance
            result = [prefix + i for i in result]
        except Exception as exc:
            print "Error: %s" % (exc, )

        return result

    def check_iss996(self):
        print '# check_iss996'
        result = []
        self.iss996_visited_dirs = set()

        for i in self.instances:
            instance_result = self.check_instance_iss996(i)
            result.extend(instance_result)

        # search orphan dirs
        all_instance_dirs = set(os.listdir(self.INSTANCES_DIR))
        orphan_dirs = all_instance_dirs - self.iss996_visited_dirs

        for filename in sorted(orphan_dirs):
            result.append('orphan instance: %s' % (filename, ))

        result = ['ISS-996: ' + x for x in result]
        if result:
            self.print_result(result)
        else:
            print 'ISS-996: ok'


def main():
    checker = IssAgentChecks()
    checker.do_check()


if __name__ == '__main__':
    main()
