#!/usr/bin/env python2
# coding=utf-8
import json
import os
import socket
import sys
import time
import urllib2


ISS_AGENT_PORT = 25536
PREPARED = 'PREPARED'
DEBUG = False


def console(message, eol=True):
    if eol:
        suffix = '\n'
    else:
        suffix = ''

    sys.stdout.write(message + suffix)
    sys.stdout.flush()

info = console


def debug(message, eol=True):
    if DEBUG:
        console(message, eol)


def error(message):
    console('ERROR: ' + message)


def warning(message):
    console('WARNING: ' + message)


def get_info_by_http(url, tries=3):
    full_url = 'http://%s:%d%s' % (socket.getfqdn(), ISS_AGENT_PORT, url)

    for try_number in range(1, tries + 1):
        try:
            return urllib2.urlopen(full_url, timeout=5).read()
        except urllib2.HTTPError as exc:
            info("[HTTPError=%s]" % exc.code, eol=False)
            time.sleep(0.1)
            if try_number >= tries:
                break
        except socket.error as exc:
            info("[socket=%s]" % (exc, ), eol=False)
            time.sleep(0.1)
            if try_number >= tries:
                break
        except Exception as exc:
            info("[Exc=%s]" % (exc, ), eol=False)
            time.sleep(0.1)
            if try_number >= tries:
                break

    return None


def load_json(url):
    """
    load json data and transform unicode to string

        http://stackoverflow.com/questions/956867/
        http://stackoverflow.com/a/6633651/116373
    """
    raw_data = get_info_by_http(url)
    # info("<%s>" % type(raw_data), eol=False)
    try:
        return json.loads(raw_data)
    except Exception as exc:
        info("(Exception:%s=%s)" % (exc.__class__.__name__, exc), eol=False)
        return None


def resume_instances_on_localhost():
    """
    [local] iss01e.search.yandex.net > attach()
    Attached

    [global] iss01e.search.yandex.net > attach()
    Already attached
    """
    response = load_json('/attach')

    if not response:
        warning('unable to attach agent')
        return False

    info("attaching agent to cacher ", eol=False)

    while True:
        is_global = load_json('/is/global')
        if not is_global:
            info(".", eol=False)
            time.sleep(1)
        else:
            break

    info("done")


def format_instance(i):
    return "%(cfg)s (%(svc)s): current=%(current)s target=%(target)s" % \
           dict(cfg=i['configurationId'],
                svc=i['slot'].split('@', 1)[0],
                current=i['currentState'],
                target=i['targetState'])


def format_instances(instances):
    instances.sort(key=lambda x: x['slot'])
    return [format_instance(i) % i for i in instances]


def detach_agent():
    """
    call /detach
    check /is/local is true
    """
    info("detaching agent from cacher ", eol=False)

    while True:
        response = load_json('/detach')

        if response is None:
            info('.', eol=False)
            time.sleep(1)
        else:
            break

    while True:
        is_local = load_json('/is/local')
        if not is_local:
            info(".", eol=False)
            time.sleep(1)
        else:
            break

    info("done")


def stop_instances_on_localhost():
    """

    get active instances from /instances/active
    call /instance/pause/:id
    wait for empty /instances/active
    """
    detach_agent()

    apply_called = False
    start = time.time()
    prev_wrong_dump = []

    while True:
        instances = load_json('/instances')

        if instances is None:
            time.sleep(1)
            continue

        wrong_instances = [i for i in instances if not (i['currentState'] == PREPARED and i['targetState'] == PREPARED)]
        wrong_dump = format_instances(wrong_instances)

        if not wrong_instances:
            info("")
            break

        if not apply_called:
            apply_called = True
            info("call pause on running instances")
            for i in wrong_instances:
                stop_url = '/instance/pause/%s' % i['slot']
                load_json(stop_url)

        if wrong_dump != prev_wrong_dump:
            if DEBUG:
                info("")  # Enter

            for i in wrong_dump:
                info(i)
            prev_wrong_dump = wrong_dump

            debug('%d instances not in PREPARED state (%d sec passed)' %
                  (len(wrong_instances), int(time.time() - start)))
        else:
            debug(".", eol=False)

        time.sleep(3)

    info('all instances are stopped')


def help_and_exit():
    sys.exit("usage: %s <stop|resume|start> [-v]" % os.path.basename(sys.argv[0]))


def main():
    global DEBUG

    start = time.time()
    args = sys.argv[1:]

    if not args:
        help_and_exit()

    if '-v' in args:
        DEBUG = True

    if 'stop' in args:
        stop_instances_on_localhost()
    elif 'resume' in args or 'start' in args:
        resume_instances_on_localhost()
    else:
        help_and_exit()

    console('# elapsed time: %d sec' % int(time.time() - start))


if __name__ == '__main__':
    main()
