import sys
import os
import json
import argparse
import re
import hashlib
import gevent
import glob
import socket
import time
import getpass
import shlex

from collections import OrderedDict

import subprocess
from subprocess import Popen, PIPE

from urllib.request import Request
from urllib.request import urlopen
from urllib.request import URLError, HTTPError


class HeadRequest(Request):
    def get_method(self):
        return "HEAD"


class Colorize(object):
    def __init__(self, color=False):
        self.colormap = {}
        self.colorlist = ('VIOLET', 'BLUE', 'GREEN', 'YELLOW', 'RED', 'ENDC')
        if color:
            self.colormap['VIOLET'] = '\033[95m'
            self.colormap['BLUE'] = '\033[94m'
            self.colormap['GREEN'] = '\033[92m'
            self.colormap['YELLOW'] = '\033[93m'
            self.colormap['RED'] = '\033[91m'
            self.colormap['ENDC'] = '\033[0m'
        else:
            for color in self.colorlist:
                self.colormap[color] = ""

    def colorize(self, color, string):
        return "{}{}{}".format(self.colormap[color],
                               string,
                               self.colormap['ENDC'])

class PodInstance(object):
    def __init__(self, instance):
        self.currentStatus = instance['currentStatus']
        self.targetState = instance['targetState']
        self.podSetId = instance['podSetId']
        self.id = instance['id']
        self.host = instance['host']

class IssInstance(object):
    def __init__(self, instance):
        self.ACTIVE = 'ACTIVE'
        self.PREPARED = 'PREPARED'
        self.REMOVED = 'REMOVED'
        self.currentState = instance['currentState']
        self.targetState = instance['targetState']
        if self.currentState == self.targetState:
            self.synced = True
        else:
            self.synced = False
        self.configurationId = instance['configurationId']
        self.slot = instance['slot']
        self.instanceDir = instance['instanceDir']
        if 'properties/topology' in instance['instanceData']:
            self.topology = instance['instanceData']['properties/topology']
        else:
            self.topology = None

        if 'properties/BSCONFIG_IPORT' in instance['instanceData']:
            self.iport = instance['instanceData']['properties/BSCONFIG_IPORT']
        else:
            self.iport = None

        if 'properties/BSCONFIG_SHARDDIR' in instance['instanceData']:
            self.sharddir = \
                instance['instanceData']['properties/BSCONFIG_SHARDDIR']
        else:
            self.sharddir = None

        if 'properties/tags' in instance['instanceData']:
            self.tags = instance['instanceData']['properties/tags']
        elif 'properties/all-tags' in instance['instanceData']:
            self.tags = instance['instanceData']['properties/all-tags']
        else:
            self.tags = None

        self.pids = []

    def get_auto_tags(self):
        auto_tag = re.compile("a_(ctype|dc|geo|itype|prj|tier|topology)_.+")
        self.auto_tags = ""
        for tag in self.tags.split():
            if auto_tag.match(tag):
                self.auto_tags += tag + " "
        self.auto_tags += "\n"
        if self.auto_tags is not None:
            return True
        else:
            return False

    def get_sky_tags(self):
        sky_tag = re.compile("a_(ctype|dc|geo|itype|prj|tier)_.+")
        sky_tags_list = []
        self.sky_tags = ""
        for tag in self.tags.split():
            if sky_tag.match(tag):
                sky_tags_list.append(tag)

        for tag in sky_tags_list:
            if tag is not sky_tags_list[-1]:
                self.sky_tags += "I@{} . ".format(tag)
            else:
                self.sky_tags += "I@{}\n".format(tag)

        if self.sky_tags is not None:
            return True
        else:
            return False

    def fuser(self, filename):
        fusercmd = '/usr/bin/sudo -u loadbase /bin/fuser ' + filename
        out, err = Popen(
                        fusercmd,
                        shell=True,
                        stdout=PIPE,
                        stderr=PIPE
                        ).communicate()
        return out.strip(' ')

    def getpids(self):
        for f in glob.glob(self.instanceDir + "/state/*.lock"):
            self.pids.append(self.fuser(f))

    def ps(self):
        out = ''
        counter = 0
        pscmd = '/bin/ps --pid '
        if not 'ISS_HELPER_PS_STRING' in os.environ:
            psstring = 'user:10,pid,nlwp,ni,pri,psr,pcpu,pmem,'
            psstring += 'vsz:8,rss:8,stat,wchan:14,start,time,'
            psstring += 'command'
        else:
            psstring = os.environ['ISS_HELPER_PS_STRING']
        psargs = ' -o ' + psstring + ' ww'

        for pid in self.pids:
            if pid == '':
                continue
            cmd = pscmd + pid + psargs
            pout, err = Popen(
                            cmd,
                            shell=True,
                            stdout=PIPE,
                            stderr=PIPE
                             ).communicate()
            if counter == 0:
                out += pout
            else:
                out += pout.split('\n')[1]
                out += '\n'
            counter += 1
        return out


class IssHost(object):
    def __init__(self,
                iss_host="localhost",
                iss_port="25536",
                iss_json_path="instances",
                dump=False,
                filename=None,
                ):
        self.iss_host = iss_host
        self.iss_port = iss_port
        if not filename:
            request = Request('http://%s:%s/%s' %
                            (iss_host, iss_port, iss_json_path))

            try:
                response = urlopen(request, timeout=20)
            except URLError as e:
                print(str(e.reason))
                sys.exit(1)
            except socket.timeout as e:
                print(str(e.reason))
                sys.exit(1)

            iss_json = response.read()
            self.host_json = json.loads(iss_json)

            if dump:
                dump_path = "/var/tmp/ih-debug-dump-" + \
                                getpass.getuser() + "-" +\
                                str(int(time.time()))
                with open(dump_path, 'w+') as f:
                    f.write(iss_json)
        else:
            with open(filename) as f:
                self.host_json = sorted(json.loads(f.read()), key = lambda item: item['slot'])

    def get_instance_list(self, condfunc):
        instance_list = []
        for inst in self.host_json:
            instance = IssInstance(inst)
            if condfunc(instance):
                instance_list.append(instance)

        return instance_list

    def get_pod_instance_list(self, condfunc):
        request = Request('http://%s:%s/%s/%s' %
                                  (self.iss_host, self.iss_port, "pods", "status"))

        try:
            response = urlopen(request, timeout=20)
        except URLError as e:
            print(str(e.reason))
            sys.exit(1)
        except socket.timeout as e:
            print(str(e.reason))
            sys.exit(1)

        result = []
        iss_json = response.read()

        for conf in json.loads(iss_json)["pod_status_info"]:
            pod_instance = PodInstance(conf)
            if condfunc(pod_instance):
                result.append(pod_instance)
        return result

class IssDumpJson(object):
    def __init__(self,
                instance_dir
                ):
        self.dump_json = 'dump.json'
        self.instance_dir = instance_dir + "/"
        self.dump_json_path = self.instance_dir + self.dump_json
        if not os.path.isfile(self.dump_json_path):
            self.removed = True
            return
        else:
            self.removed = False
        with open(self.dump_json_path) as dj_file:
            self.instance_json = json.loads(dj_file.read())

        self.resources_list = self.instance_json['resources'].keys()

    def get_res_urls(self, name):
        return self.instance_json['resources'][name]['urls']

    def get_res_md5(self, name):
        checksum = \
            self.instance_json['resources'][name]['verification']['checksum']
        if checksum != "EMPTY:":
            return checksum.split(':')[1]
        else:
            return None


class IssResource(object):
    def __init__(self, name, path, urls, md5):
        self.name = name
        self.path = path
        self.urls = urls
        self.urls_checked = None
        self.IssMd5 = md5
        self.isdir = False
        if not self.IssMd5 and os.path.isdir(self.path):
            self.isdir = True
            self.sky_dirstate = self.get_dir_state()
        self.CurMd5 = None

    def md5sum(self, filename, blocksize=65536):
        hash = hashlib.md5()
        with open(filename, "rb") as f:
            for block in iter(lambda: f.read(blocksize), ""):
                hash.update(block)

        return hash.hexdigest()

    def get_dir_state(self):
        out = {}
        for url in self.urls:
            sky_json = self.skyfiles(url)
            for fl in sky_json:
                path = self.name + "/" + fl['name']
                if 'md5sum' in fl:
                    out[path] = fl['md5sum']
        return out

    def check_res_md5(self, check_dir_md5):
        if self.isdir:
            self.real_dirstate = {}
            if check_dir_md5:
                for fl in self.sky_dirstate.keys():
                    path = self.path + "/" + fl
                    #self.real_dirstate[fl] = self.md5sum(path)
            self.CurMd5 = None
            return self.CurMd5
        else:
            self.CurMd5 = self.md5sum(self.path)
            return self.CurMd5

    def skyfiles(self, url):
        try:
            command_line = "sky files --json " + url
            args = shlex.split(command_line)
            p = Popen(args, stdout=subprocess.PIPE)
            p.wait()
            j = p.stdout.read()
            return json.loads(j.strip())
        except Exception as err:
            return err

        return rbtorrent_inf

    def check_skyfiles(self, url):
        try:
            command_line = "sky files --json " + url
            args = shlex.split(command_line)
            p = Popen(args, stdout=subprocess.PIPE)
            p.wait()
            j = p.stdout.read()
            json.loads(j.strip())
        except Exception as err:
            return (url, err)

        return (url, 200)

    def http_head(self, url):
        try:
            response = urlopen(HeadRequest(url), timeout=10)
        except HTTPError as err:
            return (url, err.code)
        except Exception as err:
            return (url, err)

        return (url, response.getcode())

    def check_urls(self):
        rbtorrent_expr = re.compile("^rbtorrent:.+")
        http_expr = re.compile("^http(s)?:.+")

        for url in self.urls:
            self.urls_checked = {}
            if rbtorrent_expr.match(url):
                result = self.check_skyfiles(url)
                self.urls_checked[result[0]] = result[1]
                return self.urls_checked
            elif http_expr.match(url):
                result = self.http_head(url)
                self.urls_checked[result[0]] = result[1]
                return self.urls_checked


class PrintIssState(object):
    def __init__(self):
        self.cl = Colorize(sys.stdout.isatty())

    def gen_instance_state_str(self, instance):
        if instance.synced:
            if instance.currentState == instance.ACTIVE:
                return self.cl.colorize('GREEN', instance.ACTIVE)
            elif instance.currentState == instance.PREPARED:
                return self.cl.colorize('BLUE', instance.PREPARED)
            else:
                return instance.currentState
        else:
            return self.cl.colorize(
                                   'RED',
                                    "{} => {}".format(
                                                instance.currentState,
                                                instance.targetState
                                                     )
                                   )

    def gen_instance_name(self, instance):
        if instance.iport:
            return "{}@{}".format(instance.iport, instance.configurationId)
        else:
            return "NO_PORT@{}".format(instance.configurationId)

    def prepare_instances_info_string_short(self, instance):
        instance_md5 = re.findall('[_0-9a-zA-Z]+$', instance.instanceDir)[0]
        return "{}: {}  [{}]\n".format(
                                    self.gen_instance_name(instance),
                                    instance_md5,
                                    self.gen_instance_state_str(instance)
                                     )

    def prepare_instances_info_string(self, instance):
        output = ""
        output += "{} :\n".format(instance.instanceDir)
        output += "    Id: {}  [{}]\n".format(
                                        instance.configurationId,
                                        self.gen_instance_state_str(instance)
                                             )
        if instance.topology:
            output += "    Topology: {}\n".format(instance.topology)
        if instance.iport:
            output += "    Port: {}\n".format(instance.iport)
        if instance.sharddir:
            output += "    IndexDir: {}\n".format(instance.sharddir)
        output += "\n"
        return output

    def gen_instances_info_list(self, instance_list, short=False):
        out = ""
        if short:
            prepare_func = self.prepare_instances_info_string_short
        else:
            prepare_func = self.prepare_instances_info_string

        for instance in instance_list:
            out += prepare_func(instance)

        return out

    def gen_pod_instances_info_list(self, pod_instance_list):
        out = ""
        template = "{}:\n    Pod set id: {} ID: {}  Target state: [{}]  Status: [{}]\n\n"
        for instance in pod_instance_list:
            if instance.targetState == 'ACTIVE':
                targetState = self.cl.colorize('GREEN', "ACTIVE")
            else:
                targetState = self.cl.colorize('RED', instance.targetState)

            if instance.currentStatus == 'READY':
                currentStatus = self.cl.colorize('GREEN', "READY")
            elif instance.currentStatus == 'IN_PROGRESS':
                currentStatus = self.cl.colorize('BLUE', "IN_PROGRESS")
            else:
                currentStatus = self.cl.colorize('RED', instance.currentStatus)

            out += template.format(instance.host, instance.podSetId, instance.id, targetState, currentStatus)

        return out

    def gen_res_info(self, res, gen_dir_md5=False):
        out = ""

        if res.urls_checked is None:
            for url in res.urls:
                out += "            Url: {}\n".format(url)
        else:
            for url in res.urls_checked.keys():
                if res.urls_checked[url] is 200:
                    msg = self.cl.colorize('GREEN', "OK")
                    out += "            Url: {} [{}]\n".format(url, msg)
                else:
                    msg = self.cl.colorize(
                                'RED',
                                "Response: {}".format(res.urls_checked[url])
                                          )
                    out += "            Url: {} [{}]\n".format(url, msg)

        if res.isdir and gen_dir_md5:
            out += "            Directory:\n"
            for fl in res.sky_dirstate.keys():
                out += "                {}: {}\n".format(
                                                        fl,
                                                        res.sky_dirstate[fl]
                                                        )
        elif res.isdir:
            out += "            Md5: Directory\n"

        else:
            if res.CurMd5 is None:
                out += "            Md5: {}\n".format(res.IssMd5)
            elif res.CurMd5 == res.IssMd5:
                msg = self.cl.colorize('GREEN', "OK")
                out += "            Md5: {} [{}]\n".format(res.IssMd5, msg)
            elif res.CurMd5 != res.IssMd5:
                msg = self.cl.colorize(
                                    'RED',
                                    "Current Md5: {}".format(res.CurMd5)
                                      )
                out += "            Md5: {} [{}]\n".format(res.IssMd5, msg)

        return out

    def gen_resources_stat(
                        self,
                        instance,
                        check_md5=False,
                        check_url=False,
                        check_dir_md5=False,
                          ):
        out = ""
        out += "    Resources:\n"
        dumped_state = IssDumpJson(instance.instanceDir)
        if dumped_state.removed is True:
            if instance.currentState == instance.REMOVED:
                out += "        Not available: "
                out += "instance current state is {}".format(instance.REMOVED)
                return out
            elif instance.targetState == instance.REMOVED:
                out += "        Not available: "
                out += "instance target state is {}".format(instance.REMOVED)
                return out

        resources = {}

        for res in dumped_state.resources_list:
            full_path = dumped_state.instance_dir + res
            resources[res] = IssResource(
                                name=res,
                                path=full_path,
                                urls=dumped_state.get_res_urls(res),
                                md5=dumped_state.get_res_md5(res),
                                        )

        if check_md5 or check_url:
            if check_md5 and not check_url:
                job_fc = (
                    lambda resources, res:
                        resources[res].check_res_md5(check_dir_md5)
                         )
            elif not check_md5 and check_url:
                job_fc = (lambda resources, res: resources[res].check_urls())
            elif check_md5 and check_url:
                job_fc = (
                        lambda resources, res:
                            [resources[res].check_res_md5(check_dir_md5),
                            resources[res].check_urls()]
                         )
            jobs = {}
            for res in resources.keys():
                jobs[res] = gevent.spawn(job_fc, resources, res)
            gevent.joinall(jobs.values(), timeout=10)

        for res in resources.keys():
            out += "        {}\n".format(res)
            out += self.gen_res_info(resources[res], check_dir_md5)

        return out

    def gen_tag_list(self, instance_list, print_func):
        out = ""
        for instance in instance_list:
            out += self.cl.colorize(
                                    'BLUE',
                                    "{}@{}\n".format(
                                                    instance.iport,
                                                    instance.configurationId
                                                    )
                                   )
            out += print_func(instance)
        return out

def main():
    parser = argparse.ArgumentParser(description='Iss helper. \
                 Wiki: https://wiki.yandex-team.ru/users/noiseless/isshelper')
    parser.add_argument('--dump', '-d', action="store_true",
                help="Save iss json to /var/tmp")
    parser.add_argument('--file', '-f', action="store",
                help="Try to load iss json from file")
    subparsers = parser.add_subparsers(dest="subparser_name")
    list_parser = subparsers.add_parser('list', help='List instances on the host')
    list_parser.add_argument('--active', '-a', action="store_true",
                help="Show only active instances")
    list_parser.add_argument('--prepared', '-p', action="store_true",
                help="Show only prepared instances")
    list_parser.add_argument('--unsync', '-u', action="store_true",
                help="Show instances, which currentState != targetState")
    list_parser.add_argument('--short', '-s', action="store_true",
                help="Show instances list only")
    list_parser.add_argument('expression', action="store", nargs='?',
                help="Show only instances which are matched by expression")
    listtags_parser = subparsers.add_parser('listtags',
                help='List tags of active instances')
    listtags_parser.add_argument('--auto', '-a', action="store_true",
                help="Show only autotags")
    listtags_parser.add_argument('--sky', '-s', action="store_true",
                help="Show autotags in skynet format")
    listtags_parser.add_argument('expression', action="store", nargs='?',
                help="Show last active instances matched by expression")
    show_parser = subparsers.add_parser('show',
                help='Show instance properties and resources')
    show_parser.add_argument('expression', action="store",
                help="Show only instances which instance hash\
                    are matched by expression")
    show_parser.add_argument('--md5', '-m', action="store_true",
                help="Check md5 sums of resources")
    show_parser.add_argument('--links', '-l', action="store_true",
                help="Check urls of resources")
    show_parser.add_argument('--conf', '-c', action="store_true",
                help="Match by configuration id (only active instances)")
    show_parser.add_argument('--unprepared', '-u', action="store_true",
                help="Match by configuration id (only unprepared instances)")
    show_parser.add_argument('--dirs', '-d', action="store_true",
                help="Check md5 for files in directories (shards)")
    ps_parser = subparsers.add_parser('ps',
                help='Show instance processes')
    ps_parser.add_argument('expression', action="store",
                help="Show only processes of instances which instance hash\
                    are matched by expression")
    ps_parser.add_argument('--conf', '-c', action="store_true",
                help="Match by configuration id (only active configurations)")

    args = parser.parse_args()
    if args.file:
        iss_host = IssHost(dump=False, filename=args.file)
    elif args.dump:
        iss_host = IssHost(dump=True)
    else:
        iss_host = IssHost(dump=False)
    host_state = PrintIssState()
    instance_dir_regex = re.compile("{}".format(args.expression))

    if args.subparser_name == 'list':
        if args.expression:
            match_cfg = (lambda expr, inst: re.findall(expr, inst.configurationId))
        else:
            match_cfg = (lambda x, y: True)

        if args.unsync:
            is_synced = (lambda inst: not inst.synced)
        else:
            is_synced = (lambda x: True)

        if args.active and args.prepared:
            condfunc = (lambda inst:
                                (inst.targetState == inst.ACTIVE
                                or
                                inst.targetState == inst.PREPARED)
                                and
                                is_synced(inst)
                                and
                                match_cfg(args.expression, inst)
                   )
        elif args.active:
            condfunc = (lambda inst:
                                inst.targetState == inst.ACTIVE
                                and
                                is_synced(inst)
                                and
                                match_cfg(args.expression, inst))
        elif args.prepared:
            condfunc = (lambda inst:
                                inst.targetState == inst.PREPARED
                                and
                                is_synced(inst)
                                and
                                match_cfg(args.expression, inst))
        else:
            condfunc = (lambda inst:
                                is_synced(inst)
                                and
                                match_cfg(args.expression, inst))

        print(host_state.gen_instances_info_list(
                                        iss_host.get_instance_list(condfunc),
                                        args.short
                                            ).rstrip('\n'))
        print('\n')
        print(host_state.gen_pod_instances_info_list(
                                            iss_host.get_pod_instance_list(condfunc),
                                        ).rstrip('\n'))


    elif args.subparser_name == 'listtags':
        if args.expression:
            match_cfg = (lambda expr, inst: re.findall(expr, inst.configurationId))
        else:
            match_cfg = (lambda x, y: True)

        condfunc = (lambda inst:
                            inst.targetState == inst.ACTIVE
                            and
                            inst.synced
                            and
                            match_cfg(args.expression, inst))
        print_func = (lambda inst: inst.tags + "\n")

        if args.auto:
            print_func = (lambda inst:
                                inst.auto_tags if inst.get_auto_tags()
                                else "No autotags for this instance!")
        elif args.sky:
            print_func = (lambda inst:
                                inst.sky_tags if inst.get_sky_tags()
                                else "No autotags for this instance!")
        print(host_state.gen_tag_list(
                                iss_host.get_instance_list(condfunc),
                                print_func
                                 ).rstrip("\n"))

    elif args.subparser_name == 'show':
        condfunc = (lambda inst: instance_dir_regex.findall(inst.instanceDir))
        match_cfg = (lambda expr, inst: re.findall(expr, inst.configurationId))
        if args.conf:
            if args.unprepared:
                print("Option [--unprepared] conflicts with [--conf]")
                sys.exit(1)
            condfunc = (
                    lambda inst:
                        match_cfg(args.expression, inst)
                        and
                        inst.targetState == inst.ACTIVE
                   )
        if args.unprepared:
            if args.conf:
                print("Option [--conf] conflicts with [--unprepared]")
                sys.exit(1)
            condfunc = (
                    lambda inst:
                        match_cfg(args.expression, inst)
                        and
                        inst.targetState == inst.PREPARED
                        and
                        inst.currentState != inst.ACTIVE
                        and not
                        inst.synced
                   )

        instance_list = iss_host.get_instance_list(condfunc)
        instance_list_length = len(instance_list)
        counter = 1
        for instance in instance_list:
            print(host_state.prepare_instances_info_string(instance).rstrip('\n'))
            print("    Tags: {}".format(instance.tags))
            print(host_state.gen_resources_stat(
                                        instance,
                                        args.md5,
                                        args.links,
                                        args.dirs,
                                           ).rstrip('\n'))
            if counter < instance_list_length:
                print("")
            counter += 1

    elif args.subparser_name == 'ps':
        condfunc = (lambda inst: instance_dir_regex.findall(inst.instanceDir))
        for instance in iss_host.get_instance_list(condfunc):
            instance.getpids()
            print(instance.ps().rstrip('\n'))
        if args.conf:
            match_cfg = (lambda expr, inst: re.findall(expr, inst.configurationId))
            condfunc = (
                    lambda inst:
                        match_cfg(args.expression, inst)
                        and
                        inst.targetState == inst.ACTIVE
                   )
            for instance in iss_host.get_instance_list(condfunc):
                instance.getpids()
                print(host_state.gen_instance_name(instance) + ":")
                print(instance.ps().rstrip('\n'))

