#!/usr/bin/env python2.7
# coding=utf-8
"""
Создает JSON-фрагменты для метрик JobAPI в графите
"""
import json

A_PREFIXES = ("qloud-production.nirvana.job-service.bare-metal-prod_qloud_a.backend.*.",)
B_PREFIXES = ("qloud-production.nirvana.job-service.bare-metal-prod_qloud_b.backend.*.",)
C_PREFIXES = ("qloud-production.nirvana.job-service.bare-metal-prod_qloud_c.backend.*.",)
CC_PREFIXES = ("qloud-production.nirvana.job-service.bare-metal-prod_qloud_cc.backend.*.",)
# D_PREFIXES = ("qloud-production.nirvana.job-service.bare-metal-prod_qloud_d.backend.*.",)

#     "qloud-production.nirvana.job-service.bare-metal-prod_qloud_c.backend.*.",
PROD_PREFIXES = (
    "qloud-production.nirvana.job-service.bare-metal-production.backend.*.",
    "qloud-production.nirvana.job-service.bare-metal-prod_qloud_a.backend.*.",
    "qloud-production.nirvana.job-service.bare-metal-prod_qloud_b.backend.*.",
    "qloud-production.nirvana.job-service.bare-metal-prod_qloud_cc.backend.*.",
    # "qloud-production.nirvana.job-service.bare-metal-prod_qloud_d.backend.*.",
)


def make_group_metrics(name, title, prefixes):
    """
    name should start from "ru."
    """
    assert name.starts("ru.")
    parts = [prefix + name for prefix in prefixes]
    if len(parts) > 1:
        group_body = ", ".join(parts)
        group = "group(%s)" % group_body
    else:
        group = parts[0]

    alias = "aliasByNode(%s, 5)" % group.lower()
    result = {
        "target": [
            alias
        ],
        "title": title
    }
    return result


"""
https://graphite.qe-infra.yandex-team.ru/dashboard#dev-jobservice
https://graphite.qe-infra.yandex-team.ru/dashboard#prod-jobservice
https://graphite.qe-infra.yandex-team.ru/dashboard#prod-jobservice-a
https://graphite.qe-infra.yandex-team.ru/dashboard#prod-jobservice-b
https://graphite.qe-infra.yandex-team.ru/dashboard#prod-jobservice-c


https://wiki.yandex-team.ru/JobAPI/metrics/list/

https://st.yandex-team.ru/SCHEDULERS-807#1467968234000
"""

DEV_ROOT_NAME = "qloud-production.nirvana.job-service.bare-metal-dev_sas.backend.*.ru.yandex.schedulers.groups"
DEV_SYSTEM_NAME = "qloud-production.nirvana.job-service.bare-metal-dev_sas.backend.*.ru.yandex.schedulers"

METRICS_DATA = (
    ("wait-in-queue", "wait in queue"),
    ("applied-to-iss-ack-finish", "JobAPI+ISS latency (#1)"),
    ("schedule-to-iss-ack-finish", "schedule to ISS + finish (#2)"),
    ("schedule-to-started-finish", "schedule to started in Nirvana (#3)"),
    ("applied-to-iss-ack", "group starting lag"),
    ("finishing", "group finishing lag"),
    ("restart-count", "restart count x 1000"),
    # ("applied-to-started", "applied to started lag"),
)

SCHEDULERS_TO_VANILLA_FLAGS = [
    ('TOTAL', ('',)),
    ('all-or-none', ('vanilla', 'r2')),
    ('master-slave', ('vanilla', 'r2')),
    # ('client-server', ('vanilla', 'r2'))
]

JOBAPI_SYSTEM_METRICS = (
    # heartbeat:
    ("step.loop-lag", "planning loop lag, msec"),
    ("ip-broker.endpoints-cleaner.loop-lag", "ip-broker endpoint cleaner lag, msec"),
    ("group-metrics.loop-lag", "group metrics reporter lag, msec"),
    # URL notifier:
    ("url_notifier.queue_size", "notifier: queue size"),
    ("url_notifier.connect_and_response.time.p95", "notifier: connect + response time, msec"),
    ("url_notifier.connect_errors.count", "notifier: connect errors"),
    ("url_notifier.non_2xx_errors.count", "notifier: error responses num"),
    ("url_notifier.total_submits.count", "notifier: all responses num"),
    ("url_notifier.unknown_errors.count", "notifier: other errors num"),
)


def make_metrica_full_path(root, scheduler, name, vanilla):
    parts = [root, scheduler, name]
    if vanilla:
        parts.append(vanilla)
    parts.append("p95")
    path = ".".join(parts)
    return path.lower()


def create_one_dev_metrica(scheduler, name, vanilla, label):
    fullname = make_metrica_full_path(DEV_ROOT_NAME, scheduler, name, vanilla)
    return dict(title=label, target=["aliasByNode(%s, 5)" % fullname.lower()])


def create_one_dev_metrica_system(name, label):
    fullname = DEV_SYSTEM_NAME + "." + name
    return dict(title=label, target=["aliasByNode(%s, 5)" % fullname.lower()])


def format_title(scheduler, label, vanilla):
    title = scheduler + ": " + label
    if vanilla:
        title += " (%s)" % vanilla
    title += ", p95"
    return title


PROD_ROOT_NAME = "ru.yandex.schedulers.groups"
SYSTEM_ROOT_NAME = "ru.yandex.schedulers"


def dump_json_without_brackets(data, filename):
    blob = json.dumps(data, indent=2).lstrip("[").rstrip("]")
    open(filename, "wb").write(blob)


def make_all_dev_metrics(fn):
    result = []
    for scheduler, vanilla_flags in SCHEDULERS_TO_VANILLA_FLAGS:
        for vanilla in vanilla_flags:
            for name, label in METRICS_DATA:
                title = format_title(scheduler, label, vanilla)
                result.append(create_one_dev_metrica(scheduler, name, vanilla, title))

    for name, label in JOBAPI_SYSTEM_METRICS:
        result.append(create_one_dev_metrica_system(name, label))

    dump_json_without_brackets(result, fn)


def make_prod_metrics(prefixes, fn):
    result = []
    for scheduler, vanilla_flags in SCHEDULERS_TO_VANILLA_FLAGS:
        for vanilla in vanilla_flags:
            for name, label in METRICS_DATA:
                full_name = make_metrica_full_path(PROD_ROOT_NAME, scheduler, name, vanilla)
                full_label = format_title(scheduler, label, vanilla)
                result.append(make_group_metrics(full_name, full_label, prefixes))

    for name, label in JOBAPI_SYSTEM_METRICS:
        result.append(make_group_metrics(SYSTEM_ROOT_NAME + "." + name, label, prefixes))
        # full_name = SYSTEM_ROOT_NAME PROD_ROOT_NAME, None, name, vanilla)
        # full_label = format_title(scheduler, label, vanilla)
        # result.append(make_group_metrics(full_name, full_label, prefixes))
        # # make_group_metrics(name, title, prefixes)

    dump_json_without_brackets(result, fn)


make_all_dev_metrics("dev.json")

make_prod_metrics(PROD_PREFIXES, "prod-ALL.json")

make_prod_metrics(A_PREFIXES, "prod-a.json")
make_prod_metrics(B_PREFIXES, "prod-b.json")
make_prod_metrics(C_PREFIXES, "prod-c.json")
make_prod_metrics(CC_PREFIXES, "prod-cc.json")
# make_prod_metrics(D_PREFIXES, "prod-d.json")
