#!/usr/bin/env python
# -*- coding: utf-8 -*-
"""
Decorator for function profiling using standard cProfile{r}
"""
import base64
import cProfile
import logging
import marshal
import optparse
import pstats
import re
import sys
import types
import zlib


log = logging.getLogger()


def _format_profile_data(params_dict):
    params_default = {
        'func_name': '-',
        'call_id': '-',
        'profile_data': '',
    }
    params_default.update(params_dict)
    line = 'profile:%(func_name)s:%(call_id)s:%(profile_data)s' % params_default
    return line


def log_profile_event(params_dict):
    """
    stub for registry profiling system
    """
    line = _format_profile_data(params_dict)
    log.debug(line)


def print_profile_event(params_dict):
    """
    print profiling result to stdout
    """
    line = _format_profile_data(params_dict)
    print line


def get_file_profile_event_func(filename):

    def file_profile_event(params_dict):
        """
        print profiling result to stdout
        """
        line = _format_profile_data(params_dict)
        fh = open(filename, 'a')
        fh.write(line + "\n")
        fh.close()

    return file_profile_event


LOG_PROFILE_DATA_FUNC = log_profile_event


def set_profile_logger_func(newfunc):
    global LOG_PROFILE_DATA_FUNC

    assert callable(newfunc)

    prev = LOG_PROFILE_DATA_FUNC
    LOG_PROFILE_DATA_FUNC = newfunc
    return prev

# ---------------- START OF PROFILING DECORATOR --------------------------


def _get_call_id(args):
    """
    Try to get 'call_id' parameter from dictionary arguments.
    Return current timestamp if no call_id found.
    """
    for arg in args:
        if isinstance(arg, types.DictType):
            call_id = arg.get('call_id', None)
            if call_id and isinstance(call_id, types.StringType):
                return call_id

    return '-'


def _get_profile_result(prof):
    """
    extract and return function profiling data encoded in base64
    """
    prof.create_stats()
    profile_dump = marshal.dumps(prof.stats)

    try:
        profile_compressed = zlib.compress(profile_dump)
    except zlib.error:
        profile_compressed = profile_dump

    text_data = base64.encodestring(profile_compressed).replace('\n', '')

    return text_data


def _log_profile_result(func_name, call_id, prof):
    """
    Log function profiling result as base64 encoded string.
    """
    profile_data = _get_profile_result(prof)
    params_dict = {
        'func_name': func_name,
        'call_id': call_id,
        'profile_data': profile_data,
    }
    LOG_PROFILE_DATA_FUNC(params_dict)


def profileit(func):
    """
    Decorator for profiling function using cProfile module.

    This is port from:
    http://stackoverflow.com/questions/5375624/a-decorator-that-profiles-a-method-call-and-logs-the-profiling-result
    """

    def wrapper(*args, **kwargs):
        """
        internal decorator wrapper's docstring
        """
        call_id = _get_call_id(args)
        prof = cProfile.Profile()
        retval = prof.runcall(func, *args, **kwargs)
        _log_profile_result(func.__name__, call_id, prof)
        return retval

    return wrapper

# ---------------- END OF PROFILING DECORATOR --------------------------

#PROFILING_RECORD_RE = re.compile(r'<profileFunction\s+(\S+):([^>]*)>([^<\s]+)' + re.escape('</profileFunction>'))
PROFILING_RECORD_RE = re.compile(r'profile:([^:]*):([^:]*):(\S+)\s*$')

# profile:%(func_name)s:%(call_id)s:%(profile_data)s


def parse_log_string(logline):
    """
    sample log string parsing routine
    """
    match = PROFILING_RECORD_RE.search(logline)

    if match:
        function, call_id, text_data = match.group(1, 2, 3)
        return function, call_id, text_data

    return None


LOG_LINE_PARSE_FUNC = parse_log_string


def set_log_line_parse_func(newfunc):
    global LOG_LINE_PARSE_FUNC

    assert callable(newfunc)

    prev = LOG_LINE_PARSE_FUNC
    LOG_LINE_PARSE_FUNC = newfunc
    return prev


def get_next_stat(fh):
    """
    load profileFunction clause from text file handle
    """
    for rawline in fh:
        logline = rawline.splitlines()[0]
        res = LOG_LINE_PARSE_FUNC(logline)

        if res and len(res) == 3:
            return res
        else:
            continue

    return None, None, None


def add_stats(main_stat, another_stat):
    """
    add another_stat to main_stat
    return main_stat
    """
    if main_stat is None:
        return another_stat
    else:
        main_stat.add(another_stat)
        return main_stat


class FakeStatSource(object):
    """
    take stats in constructor.
    provide .stats member with it
    have create_stats() to fool pstats.Stats.load_stats()
    """
    def __init__(self, stats):
        self.stats = stats

    def create_stats(self):
        pass


def assemble_log_stats(in_logfile, out_stat_file, function=None, call_id=None):

    def stat_filter(func_, call_id_):

        if function and func_ != function:
            return False

        if call_id and call_id_ != call_id:
            return False

        return True

    main_stat = None

    fh = open(in_logfile, 'r')

    while True:
        func_, call_id_, text_data = get_next_stat(fh)
        log.debug('get stat for function(%s):call_id(%s)' % (func_, call_id_))

        if not func_:
            break

        if not stat_filter(func_, call_id_):
            continue

        try:
            bin_data = base64.decodestring(text_data)

            try:
                bin_data = zlib.decompress(bin_data)
            except zlib.error:
                pass

            stat_data = marshal.loads(bin_data)
            stat_source = FakeStatSource(stat_data)
            stat_piece = pstats.Stats(stat_source)
            #stat_piece.load_stats(stat_source)
        except:
            log.exception('error decode line: func_=%r call_id_=%r text_data=%r' % (func_, call_id_, text_data))
            continue

        # update stats with new data
        main_stat = add_stats(main_stat, stat_piece)

    fh.close()

    log.debug('main_stat: %r' % main_stat)

    if main_stat:
        main_stat.dump_stats(out_stat_file)
    else:
        print >> sys.stderr, "no profile data found"


def main():
    """
    extract stats data from passed log
    """
    import aspn

    aspn.qs()

    parser = optparse.OptionParser(usage="[options] input output")
    parser.add_option('--function', action="store", type='string', dest="function", default=None, help="filter results by function name")
    parser.add_option('--call-id', action="store", type='string', dest="call_id", default=None, help="filter results by call_id")

    options, args = parser.parse_args()
    assert len(args) >= 1, "at least source file must be specified"

    if len(args) == 1:
        args.append(args[0] + '.stats')

    assemble_log_stats(args[0], args[1], function=options.function, call_id=options.call_id)


"""
TODO:
- format periods as 1d5h8m
- profile it!
- auto connect older logs until group creation time occured
- log file inodes!

"""

if __name__ == '__main__':
    main()
