# coding=utf-8
"""
Thrift client

copy and paste from agent/configuration_download_policy_factory.py
"""
import logging
import types
import urlparse

from thrift.transport import TSocket
from thrift.transport import TTransport
from thrift.protocol import TBinaryProtocol
from thrift.Thrift import TType

from iss_thrift3.IssService import Client

log = logging.getLogger(__name__)
DEFAULT_CACHER_URL = 'thrift://iss-dev.yandex-team.ru:9090'
ADD_CLASS_INFO = False


class ThriftServerProxy(object):

    def __init__(self, url=None, timeout=10):
        if not url:
            url = DEFAULT_CACHER_URL
        self.url = url
        self.timeout = timeout

    @staticmethod
    def _create_socket(host, port):
        return TSocket.TSocket(host, int(port))

    @staticmethod
    def __get_port(parsing_result, default_port):
        if (parsing_result.port is None) or (str(parsing_result.port) == ""):
            return default_port

        return parsing_result.port

    def _create_transport(self, url=None, timeout=None):
        if not timeout:
            timeout = self.timeout

        parsing_result = urlparse.urlparse(url)
        host = parsing_result.hostname
        port = self.__get_port(parsing_result, 9090)
        thrift_socket = self._create_socket(host, port)
        thrift_socket.setTimeout(timeout * 1000)  # convert sec to ms
        transport = TTransport.TFramedTransport(thrift_socket)
        return transport

    @staticmethod
    def _create_client(transport):
        protocol = TBinaryProtocol.TBinaryProtocol(transport)
        client = Client(protocol)
        return client

    def __execute_request(self, method_name, *args):
        transport = self._create_transport(self.url)
        client = self._create_client(transport)

        log.debug('')
        log.debug('call Cacher.%s(%s)' % (method_name, ", ".join(repr(s) for s in args)))

        try:
            transport.open()
            response = getattr(client, method_name, None)(*args)
            log.debug('  response: %r' % (response, ))
            return response
        finally:
            transport.close()

    def __getattr__(self, method_name):

        def caller(*args):
            return self.__execute_request(method_name, *args)

        return caller


def get_obj_class(obj, user_ttypes_module):
    """
    assume that there are no inheritance
    """
    for name in dir(user_ttypes_module):
        if name.startswith('_'):
            continue
        member = getattr(user_ttypes_module, name, None)
        if not isinstance(member, types.ClassType):
            continue

        if isinstance(obj, member):
            return member

    raise ValueError("no class found for %r" % (obj, ))


def thrift_to_python(obj, user_ttypes_module, add_class_info=ADD_CLASS_INFO):
    """
    converts obj to python data structures according to ttypes.
    add information about classes into resulting dictionaries

    used by ISS3:
      ['MAP', 'SET', 'STRING', 'LIST', 'I64', 'BOOL', 'I32', 'STRUCT']
    """
    if not isinstance(obj, types.InstanceType):
        return obj

    obj_class = get_obj_class(obj, user_ttypes_module)
    result = {}
    system_info = {
        'class_name': obj_class.__name__,
        'field_order': []
    }

    # iterating class fields
    thrift_spec = getattr(obj_class, 'thrift_spec')
    fields = [i for i in thrift_spec if i]

    for field in fields:
        field_index = field[0]
        field_type = field[1]
        field_name = field[2]
        system_info['field_order'].append((field_index, field_name))

        value_raw = getattr(obj, field_name)

        if field_type in [TType.STRING, TType.BOOL, TType.BYTE, TType.I08,
                          TType.I16, TType.I32, TType.I64, TType.DOUBLE,
                          TType.UTF7, TType.UTF8, TType.UTF16]:
            value = value_raw
        elif field_type == TType.MAP:
            value = {}
            for k, v in value_raw.iteritems():
                k_py = thrift_to_python(k, user_ttypes_module, add_class_info)
                v_py = thrift_to_python(v, user_ttypes_module, add_class_info)
                value[k_py] = v_py
        elif field_type in [TType.SET, TType.LIST]:
            value = [thrift_to_python(i, user_ttypes_module, add_class_info) for i in value_raw]
            if field_type == TType.SET:
                value = set(value)
        elif field_type == TType.STRUCT:
            value = thrift_to_python(value_raw, user_ttypes_module, add_class_info)
        elif field_type == TType.VOID:
            value = None  # experimental
        elif field_type == TType.STOP:
            raise ValueError('got thrift.Thrift.TType.STOP!')  # experimental
        else:
            raise ValueError('unsupported type: %r' % (field_type, ))

        result[field_name] = value

    if add_class_info:
        result['_system_info'] = system_info

    return result
