import os
import requests
import requests_unixsocket
import time
from requests.exceptions import ConnectionError, HTTPError

try:
    from urllib import quote_plus
except ImportError:
    from urllib.parse import quote_plus

import logging
from iss.common.support.wait import wait_for_assertion, DEFAULT_AWAIT_TIMEOUT

__author__ = 'sashakruglov@yandex-team.ru'


class RestApiException(Exception):
    def __init__(self, *args, **kwargs):
        super(RestApiException, self).__init__(*args, **kwargs)


class AgentRestAPIClient(object):
    """Client for ISS Agent RESTful API

    Provides interface for methods supported in Agent API
    """
    GET_VERSION = 'version'
    GET_STATUS = 'status'
    GET_UNISTAT = 'unistat'
    KILL_AGENT = 'kill'
    GRACEFUL_STOP = 'gracefulStop'

    PODS_INFO = 'pods/info'

    GET_INSTANCE = 'instance/{id}'
    GET_INSTANCES = 'instances'
    GET_PREEMPTED_INSTANCES = 'instances/preempted'
    REMOVE_ALL_PREEMPTED_INSTANCES = 'instances/preempted/remove/all'
    REMOVE_PREEMPTED_INSTANCES_BY_FAMILY = 'instances/preempted/remove/family/{family}'
    REMOVE_PREEMPTED_INSTANCES_BY_SERVICE = 'instances/preempted/remove/service/{service}'
    REMOVE_PREEMPTED_INSTANCES_BY_CONFIGURATION_ID = 'instances/preempted/remove/configurationId/{configurationId}'
    REMOVE_PREEMPTED_INSTANCES_BY_ID = 'instances/preempted/remove/id/{slot}/{configurationId}'
    INSTANCE_FULL_STATE = 'v1/instances/{slot}/{configurationId}/last'

    GET_INSTANCE_HISTORY = 'history/{slot}/100'
    GET_INSTANCE_HISTORY_WITH_TIMEFRAME = 'history/{slot}/100?begin={begin}000&end={end}000'

    GET_ACTIVE_INSTANCES = 'instances/active'
    GET_PENDING_AGENT_PREEMPTION_INSTANCES = 'stat/preemption'
    GET_REJECTED_PAYLOADS = 'stat/rejected'

    GET_SYSTEM_LOG = 'v0/logs/system/{file_name}?offset={offset}&limit={limit}'
    GET_SYSTEM_LOG_META = 'v0/logs/system/{file_name}?meta'
    GET_SYSTEM_LOG_LIST = 'v0/logs/system/.'

    GET_WORKLOADS_LOG = 'v0/logs/workloads/{conf_id}/{slot}/{path}'
    GET_WORKLOADS_LOG_META = 'v0/logs/workloads/{conf_id}/{slot}/{path}?meta'

    GET_WORKLOADS_FILE = 'v1/files/workloads/{conf_id}/{slot}/{path}'
    GET_WORKLOADS_FILE_META = 'v1/files/workloads/{conf_id}/{slot}/{path}?meta'

    REOPEN_LOGS = 'instances/reopenlogs'

    PAUSE_SERVICE = 'service/pause/{service}'
    ACTIVATE_SERVICE = 'service/active/{service}'
    REMOVE_SERVICE = 'service/remove/{service}'

    PAUSE_CONF = 'instance/pause/{cid}'
    ACTIVATE_CONF = 'instance/active/{cid}'
    REMOVE_CONF = 'instance/remove/{cid}'

    FORCE_UPDATE = 'update/force'
    DISABLE_UPDATE = 'update/disable'
    ENABLE_UPDATE = 'update/enable'

    GET_LOGGERS = 'loggers'
    SET_ROOT_LOGGER_LEVEL = 'loglevel/{level}'
    SET_LOGGER_LOG_LEVEL = 'loglevel/{name}/{level}'

    SET_GLOBAL_MODE = 'attach'
    SET_LOCAL_MODE = 'detach'

    IS_GLOBAL_MODE = 'is/global'
    IS_LOCAL_MODE = 'is/local'

    DUMP_CONFIGURATION = 'local/dump'
    LOAD_CONFIGURATION = 'local/load'
    PATCH_CONFIGURATION = 'local/patch'

    CHECK_JOLOKIA = 'jolokia/list'

    DISABLE_SHCEDULER = "scheduler/{name}/disable"

    CONFIG = 'config'

    def __init__(self, log_path=''):
        self._create_logger(os.path.join(log_path, 'rest_api_calls.log'))
        self.raise_for_http_errors = True

    def _create_logger(self, log_path):
        logger = logging.getLogger('%s_%s' % (__name__, time.time()))
        logger.setLevel(logging.DEBUG)
        formatter = logging.Formatter('%(asctime)s %(levelname)s %(message)s')

        fh = logging.FileHandler(log_path)
        fh.setLevel(logging.DEBUG)
        fh.setFormatter(formatter)
        logger.addHandler(fh)
        self.logger = logger

    def _make_request(self, uri, method='GET', **kwargs):
        full_url = '%s/%s' % (self.api_base_url, uri)

        try:
            self.logger.info('Making %s request to %s', method, full_url)
            r = self.session.request(method, full_url, timeout=90, **kwargs)
        except ConnectionError:
            # we usually get here after killing agent via REST API call
            self.logger.warning('Connection problem. It might be OK.', exc_info=True)
            return

        if r.history:
            r.history.append(r)  # append last request to history for more adequate logging
            redirects = ['"{0.url}" {0.status_code} ({0.reason}) '.format(request) for request in r.history]
            self.logger.info('Detected redirects: %s', ' > '.join(redirects))

        content_type = r.headers['content-type'].lower()

        self.logger.info('Got response. Status code: %s Reason: %s Content-Type: %s',
                         r.status_code, r.reason, content_type)
        self.logger.debug('Content: %s', r.content)

        if self.raise_for_http_errors:
            try:
                r.raise_for_status()
            except HTTPError as e:
                raise RestApiException(e, full_url, r.content)

        if 'json' in content_type and r.content:
            try:
                return r.json()
            except ValueError as e:
                # log error and re-raise exception to fail test
                self.logger.error('Content-type is set to JSON, but content could not be decoded', exc_info=True)
                raise e

        return r.content

    def get_instance(self, instance_id):
        uri = self.GET_INSTANCE.format(id=quote_plus(instance_id))
        return self._make_request(uri)

    def get_instances(self):
        return self._make_request(self.GET_INSTANCES)

    def get_history(self, slot):
        uri = self.GET_INSTANCE_HISTORY.format(slot=slot)
        return self._make_request(uri)

    def get_last_state(self, slot, configuration_id):
        headers = {'Content-Type': 'application/json; charset=utf-8'}
        uri = self.INSTANCE_FULL_STATE.format(slot=slot, configurationId=quote_plus(configuration_id))
        return self._make_request(uri, headers=headers)

    def get_history_with_timeframe(self, slot, begin, end):
        uri = self.GET_INSTANCE_HISTORY_WITH_TIMEFRAME.format(slot=slot, begin=begin, end=end)
        return self._make_request(uri)

    def get_active_instances(self):
        return self._make_request(self.GET_ACTIVE_INSTANCES)

    def get_preempted_instance_count(self):
        return self._make_request(self.GET_PENDING_AGENT_PREEMPTION_INSTANCES)

    def get_rejected_instance_count(self):
        return self._make_request(self.GET_REJECTED_PAYLOADS)

    def get_preempted_instances(self):
        return self._make_request(self.GET_PREEMPTED_INSTANCES)

    def remove_all_preempted(self, **kwargs):
        return self._make_request(self.REMOVE_ALL_PREEMPTED_INSTANCES, **kwargs)

    def remove_preempted_by_family(self, family, **kwargs):
        return self._make_request(self.REMOVE_PREEMPTED_INSTANCES_BY_FAMILY.format(family=family), **kwargs)

    def remove_preempted_by_service(self, service, **kwargs):
        return self._make_request(self.REMOVE_PREEMPTED_INSTANCES_BY_SERVICE.format(service=service), **kwargs)

    def remove_preempted_by_configuration_id(self, configuration_id, **kwargs):
        return self._make_request(
            self.REMOVE_PREEMPTED_INSTANCES_BY_CONFIGURATION_ID.format(configurationId=quote_plus(configuration_id)),
            **kwargs)

    def remove_preempted_by_id(self, slot, configuration_id, **kwargs):
        return self._make_request(
            self.REMOVE_PREEMPTED_INSTANCES_BY_ID.format(slot=slot, configurationId=quote_plus(configuration_id)),
            **kwargs)

    def get_system_log(self, log_file, offset='', limit=''):
        uri = self.GET_SYSTEM_LOG.format(file_name=quote_plus(log_file), offset=quote_plus(str(offset)),
                                         limit=quote_plus(str(limit)))
        return self._make_request(uri)

    def get_system_logs_meta(self, log_file):
        return self._make_request(self.GET_SYSTEM_LOG_META.format(file_name=log_file))

    def get_system_logs_list(self):
        return self._make_request(self.GET_SYSTEM_LOG_LIST)

    def get_workloads_log(self, configuration_id, slot, path):
        uri = self.GET_WORKLOADS_LOG.format(conf_id=quote_plus(configuration_id), slot=quote_plus(slot),
                                            path=quote_plus(path))
        return self._make_request(uri)

    def get_workloads_log_meta(self, configuration_id, slot, path):
        uri = self.GET_WORKLOADS_LOG_META.format(conf_id=quote_plus(configuration_id), slot=quote_plus(slot),
                                                 path=quote_plus(path)
                                                 )
        return self._make_request(uri)

    def get_workloads_file_meta(self, configuration_id, slot, path):
        uri = self.GET_WORKLOADS_FILE_META.format(conf_id=quote_plus(configuration_id), slot=quote_plus(slot),
                                                 path=quote_plus(path)
                                                 )
        return self._make_request(uri)

    def get_workloads_file(self, configuration_id, slot, path):
        uri = self.GET_WORKLOADS_FILE.format(conf_id=quote_plus(configuration_id), slot=quote_plus(slot),
                                            path=quote_plus(path))
        return self._make_request(uri)

    def reopen_logs(self):
        return self._make_request(self.REOPEN_LOGS)

    def get_version(self):
        return self._make_request(self.GET_VERSION)

    def kill_agent(self):
        return self._make_request(self.KILL_AGENT)

    def graceful_stop(self):
        return self._make_request(self.GRACEFUL_STOP)

    def disable_update(self):
        return self._make_request(self.DISABLE_UPDATE)

    def enable_update(self):
        return self._make_request(self.ENABLE_UPDATE)

    def force_update(self):
        return self._make_request(self.FORCE_UPDATE)

    def pause_service(self, service):
        uri = self.PAUSE_SERVICE.format(service=quote_plus(str(service)))
        return self._make_request(uri)

    def activate_service(self, service):
        uri = self.ACTIVATE_SERVICE.format(service=quote_plus(str(service)))
        return self._make_request(uri)

    def remove_service(self, service):
        uri = self.REMOVE_SERVICE.format(service=quote_plus(str(service)))
        return self._make_request(uri)

    def pause_conf(self, conf):
        conf = conf.replace("#", "%23")
        uri = self.PAUSE_CONF.format(cid=quote_plus(str(conf)))
        return self._make_request(uri)

    def activate_conf(self, conf):
        conf = conf.replace("#", "%23")
        uri = self.ACTIVATE_CONF.format(cid=quote_plus(str(conf)))
        return self._make_request(uri)

    def remove_conf(self, conf):
        conf = conf.replace("#", "%23")
        uri = self.REMOVE_CONF.format(cid=quote_plus(str(conf)))
        return self._make_request(uri)

    def set_global_mode(self):
        return self._make_request(self.SET_GLOBAL_MODE)

    def set_local_mode(self):
        return self._make_request(self.SET_LOCAL_MODE)

    def is_global_mode(self):
        return self._make_request(self.IS_GLOBAL_MODE)

    def is_local_mode(self):
        return self._make_request(self.IS_LOCAL_MODE)

    def check_status(self):
        return self._make_request(self.GET_STATUS)

    def get_unistat(self):
        return self._make_request(self.GET_UNISTAT)

    def get_unistat_as_dict(self):
        return {item[0]: item[1] for item in self.get_unistat()}

    def get_loggers(self):
        return self._make_request(self.GET_LOGGERS)

    def set_root_logger(self, loglevel):
        uri = self.SET_ROOT_LOGGER_LEVEL.format(level=quote_plus(loglevel))
        return self._make_request(uri)

    def set_logger(self, logger, loglevel):
        uri = self.SET_LOGGER_LOG_LEVEL.format(name=quote_plus(logger), level=quote_plus(loglevel))
        return self._make_request(uri)

    def check_jolokia(self):
        return self._make_request(self.CHECK_JOLOKIA)

    def config(self):
        return self._make_request(self.CONFIG)

    def pods_info(self):
        return self._make_request(self.PODS_INFO)

    def dump_host_configuration(self):
        return self._make_request(self.DUMP_CONFIGURATION)

    def load_host_configuration(self, content):
        return self._make_request(self.LOAD_CONFIGURATION, method='POST', json=content)

    def patch_host_configuration(self, content):
        return self._make_request(self.PATCH_CONFIGURATION, method='POST', data=content)

    def disable_scheduler(self, name):
        return self._make_request(self.DISABLE_SHCEDULER.format(name=name))

    def wait_api_started(self, timeout=DEFAULT_AWAIT_TIMEOUT):
        def assert_status():
            assert self.check_status() is not None

        wait_for_assertion(assert_status, exc_type=RestApiException, timeout=timeout)


CLIENT_HEADERS = {'User-Agent': 'ISS Test User'}


class TCPAgentRestAPIClient(AgentRestAPIClient):

    def __init__(self, agent_host, webapp_port, log_path=''):
        super(TCPAgentRestAPIClient, self).__init__(log_path)
        self.session = requests.Session()
        self.session.headers.update(CLIENT_HEADERS)
        self.api_base_url = 'http://%s:%d' % (agent_host, webapp_port)


class UnixSockeRestAPIClient(AgentRestAPIClient):

    def __init__(self, socket_path, log_path=''):
        super(UnixSockeRestAPIClient, self).__init__(log_path)
        self.session = requests_unixsocket.Session()
        self.session.headers.update(CLIENT_HEADERS)
        import urllib
        quoted_path = urllib.quote(socket_path, "")
        self.api_base_url = 'http+unix://\0%s' % (quoted_path,)
