import logging
import os
import requests
import socket
import sys
from OpenSSL import crypto
from multiprocessing import Process, Queue

from iss.common.fakes.http_server_app import create_app
from iss.common.support.wait import wait_for_assertion, wait_for_assertion_failure
from iss.common.utils import is_socket_opened

try:
    import SocketServer
except ImportError:
    import socketserver as SocketServer


def make_ssl_cert(base_path, host=None, cn=None, alt_names=None):
    if host is not None:
        cn = '*.%s/CN=%s' % (host, host)
    cert, pkey = generate_ssl_pair(cn=cn, alt_names=alt_names)

    cert_file = base_path + '.crt'
    pkey_file = base_path + '.key'

    with open(cert_file, 'wb') as f:
        f.write(crypto.dump_certificate(crypto.FILETYPE_PEM, cert))
    with open(pkey_file, 'wb') as f:
        f.write(crypto.dump_privatekey(crypto.FILETYPE_PEM, pkey))

    return cert_file, pkey_file


def generate_ssl_pair(cn=None, alt_names=None):
    from random import random

    # pretty damn sure that this is not actually accepted by anyone
    if cn is None:
        cn = '*'

    cert = crypto.X509()
    cert.set_version(2)
    cert.set_serial_number(int(random() * sys.maxsize))
    cert.gmtime_adj_notBefore(0)
    cert.gmtime_adj_notAfter(60 * 60 * 24 * 365)

    subject = cert.get_subject()
    subject.CN = cn
    subject.O = 'Dummy Certificate'

    issuer = cert.get_issuer()
    issuer.CN = 'Untrusted Authority'
    issuer.O = 'Self-Signed'

    if alt_names:
        cert.add_extensions([
            crypto.X509Extension(
                "subjectAltName", False, ", ".join(alt_names)
            )
        ])

    pkey = crypto.PKey()
    pkey.generate_key(crypto.TYPE_RSA, 1024)
    cert.set_pubkey(pkey)
    cert.sign(pkey, 'md5')

    return cert, pkey


class ResourceManager(object):
    def __init__(self, log_dir, host='::', socket_family=socket.AF_INET6, ssl=False):
        self.socket = None
        self.socket_family = socket_family
        self.process = None
        self.log_dir = log_dir
        self.log_path = os.path.join(log_dir, 'http_service.log')
        self.host = host
        self.address = None
        self.ssl = ssl

    def _assert_socket_opened(self):
        assert is_socket_opened(self.socket)

    def start(self, timeout=180):
        socket_queue = Queue()

        def _start():
            original_socket_bind = SocketServer.TCPServer.server_bind

            def socket_bind_wrapper(this):
                ret = original_socket_bind(this)
                socket_queue.put(this.socket.getsockname())
                # Recover original implementation
                SocketServer.TCPServer.server_bind = original_socket_bind
                return ret

            SocketServer.TCPServer.server_bind = socket_bind_wrapper

            app = create_app(self.log_dir)

            formatter = logging.Formatter('%(created)s %(levelname)s %(funcName)s %(message)s')
            fh = logging.FileHandler(self.log_path)
            fh.setFormatter(formatter)
            app.logger.setLevel(logging.INFO)
            app.logger.addHandler(fh)

            context = None
            if self.ssl:
                context = make_ssl_cert(self.log_dir, alt_names=['DNS:localhost', 'IP:::'])
                app.logger.info('SSL is enabled using context %s' % (context, ))

            app.logger.info("Running Resource Manager on %s", self.host)
            app.run(host=self.host, port=0, threaded=True, ssl_context=context)

        self.process = Process(target=_start)
        self.process.start()
        self.socket = socket_queue.get(timeout=timeout)
        self.address = "[%s]:%s" % (self.socket[0], self.socket[1])


    def _url(self, *path):
        tail = '/'.join([str(_) for _ in path])
        schema = 'https' if self.ssl else 'http'
        return "%s://%s/%s" % (schema, self.address, tail)

    def get(self, resource_name, sleep=0, code=200):
        return self._url('get', sleep, code, resource_name)

    def put(self, resource_name, data, sleep=0, code=200):
        result = requests.post(self._url('put', resource_name), data, verify=False)
        if result.status_code != 200:
            raise Exception("uploading %s failed with code %s" % (resource_name, result.status_code))
        return self.get(resource_name, sleep, code)

    def put_resource(self, resource_name, data, sleep=0, code=200):
        url = self.put(resource_name, data, sleep, code)
        verification = "MD5:" + self._compute_md5(data)
        return RemoteResource(url, verification)

    def delete(self, resource_name):
        result = requests.post(self._url('del', resource_name), verify=False)
        if result.status_code != 200:
            raise Exception("removing %s failed with code %s" % (resource_name, result.status_code))

    def stop(self):
        if self.process is None:
            return

        self.process.terminate()
        self.process.join()

        # wait till socket will be closed
        wait_for_assertion_failure(self._assert_socket_opened)

        self.process = None

    def _compute_md5(self, data):
        from hashlib import md5
        return md5(data).hexdigest()


class RemoteResource(object):
    def __init__(self, url, verification):
        self.url = url
        self.verification = verification
