import shutil
from hashlib import md5

import os
import re
import subprocess

from io import StringIO
from iss.common.exceptions import FailedToStartAgent
from iss.common.utils import touch
from iss.common.vmoptions_parser import VMOptionsParser

from iss.common.logger import logger


class HookNames(object):
    META = "meta"
    SERVICE = "slot"
    START_HOOK = "iss_hook_start"
    INSTALL_HOOK = "iss_hook_install"
    UNINSTALL_HOOK = "iss_hook_uninstall"
    STOP_HOOK = "iss_hook_stop"
    STATUS_HOOK = "iss_hook_status"
    VALIDATE_HOOK = "iss_hook_validate"
    NOTIFY_HOOK = "iss_hook_notify"


class ConfigArgs(object):
    DEFAULT_ULIMIT = "agent.porto.defaultUlimit"
    SKYNET_PATH = "skynet.binary.path"
    HOSTNAME = "agent.hostname"
    META_PROCESS_CONTAINER = "agent.porto.metaProcessContainer"
    UPDATE_PERIOD = 'agent.sync.autoUpdatePeriod'
    SYNC_PERIOD = 'agent.sync.autoSyncPeriod'
    VOLUMES_STORAGE_TYPE = 'agent.volume_storage_type'
    FETCHER_FILES_TIMEOUT = "agent.fetcher.filesTimeout"

    PREEMPT_REMOVED_DICT = {
        "agent.preemptionStrategy": "preempt-implicitly-removed"
    }


class IssAgent(object):
    ISS_AGENT_JAR = "iss-agent.jar"
    ISS_AGENT_SH = "iss-agent.sh"
    OOM_KILLER = "oom-killer.sh"
    ISS_AGENT_VMOPTIONS = "iss-agent.vmoptions"
    ISS_AGENT_PORTO_OPTIONS = "porto.options"
    ISS_AGENT_RUNNER = ".iss-agent"

    FILES = [ISS_AGENT_JAR,
             ISS_AGENT_SH,
             ISS_AGENT_VMOPTIONS,
             ISS_AGENT_PORTO_OPTIONS,
             ISS_AGENT_RUNNER,
             OOM_KILLER]

    def __init__(self, run_dir, agent_files, export_dir):
        # agent container properties
        self.run_mode = 'porto'
        self.agent_container = None
        self.run_dir = run_dir
        self.export_dir = export_dir

        self.porto_container_prefix = md5(run_dir.encode('utf-8')).hexdigest()
        self.log_path = os.path.join(run_dir, 'iss-agent.log')
        self.self_help_log_path = os.path.join(run_dir, 'iss-agent.selfhelp.log')

        self._agent_files_path = agent_files
        self._runner_path = os.path.join(run_dir, self.ISS_AGENT_SH)
        self._vmoptions_path = os.path.join(run_dir, self.ISS_AGENT_VMOPTIONS)
        self._porto_options_path = os.path.join(run_dir, self.ISS_AGENT_PORTO_OPTIONS)
        self._app_conf = os.path.join(run_dir, 'application.conf')

        self.options_parser = VMOptionsParser()
        self.initial_startup_marker = os.path.join(run_dir, '_initial_start_marker')

    def delete_initial_startup_marker(self):
        if os.path.exists(self.initial_startup_marker):
            os.remove(self.initial_startup_marker)

    def delete_state3(self):
        state3 = os.path.join(self.run_dir, 'state3')
        state3_backup = os.path.join(self.run_dir, 'state3.backup')
        if os.path.exists(state3):
            os.remove(state3)
        if os.path.exists(state3_backup):
            os.remove(state3_backup)

    def touch_initial_startup_marker(self):
        touch(self.initial_startup_marker)

    def get_agent_pid(self):
        """Returns Agent PID

        :return: PID as string or None
        """
        extended_pattern = '%s.*iss-agent' % self.run_dir
        pid_list = self._get_process_pids(extended_pattern)
        return pid_list and pid_list[0] or None

    def get_process_cli_args(self, pid):
        """Gets and parses process cli args from /proc/<pid>/cmdline.

        :param pid: pid of process
        :return: tuple(path to process binary, dictionary of process cli args)
        """
        out = self.perform('xargs -0 < /proc/%s/cmdline' % str(pid))
        output = re.split('\s[-]+D?', out)

        proc = output[0]
        args = output[1:]

        params = {}
        for i in range(len(args)):
            opts = re.split('[=\s]', args[i])
            arg_name = opts[0]
            val = len(opts) > 1 and opts[1] or None
            params[arg_name] = val

        return proc, params

    def put_file(self, contents, dest):
        with open(dest, 'w') as fd:
            shutil.copyfileobj(contents, fd, -1)

    def perform(self, cmd):
        logger.info(cmd)

        class RunResult(object):
            def __init__(self, popen_out):
                self.stdout, self.stderr = popen_out

        p = subprocess.Popen([cmd], stdout=subprocess.PIPE, stderr=subprocess.PIPE, shell=True)
        result = RunResult(p.communicate())

        logger.info(result.stdout)
        logger.warn(result.stderr)
        return result

    def install(self):
        self.perform('mkdir -p /ssd/db/iss3/resources')
        self.perform('mkdir -p %s' % self.run_dir)
        self.perform('ln -s /ssd/db/iss3/resources %s/resources' % self.run_dir)
        self.perform('touch %s/.iss-agent.lock' % self._agent_files_path)
        self.perform('touch %s/.alive' % self.run_dir)
        self.perform('ln -s %s/.iss-agent.lock %s/.iss-agent.lock' % (self._agent_files_path, self.run_dir))
        self.perform('sed -i \'s/eq\ \"0\"/eq\ \"1\"/\' %s/iss-agent.sh' % self._agent_files_path)
        self.perform('cp -r %s/.iss-agent %s' % (self._agent_files_path, self.run_dir))
        self.perform('cp -r %s/iss-agent* %s' % (self._agent_files_path, self.run_dir))
        self.perform('ln -s %s/jre %s/jre' % (self._agent_files_path, self.run_dir))

        self.perform('touch %s/application.conf' % self.run_dir)
        self.perform('chmod +x %s' % self._runner_path)
        self.perform('chmod +x %s' % os.path.join(self.run_dir, ".iss-agent"))
        self.perform('chmod +x %s' % os.path.join(self.run_dir, "system", "agent-runner"))

    def configure(self, conf_files):
        for file_handler, file_name in conf_files:
            resource_path = os.path.join(self.run_dir, file_name)
            self.put_file(file_handler, resource_path)

        vmoptions = self.get_file_lines(self._vmoptions_path)
        self.options_parser.parse_options_list(vmoptions)

    def get_porto_options(self):
        porto_options = self.get_file_lines(self._porto_options_path)

        options = {}
        for line in porto_options.split():
            pattern = re.compile(r"^[^#][a-zA-Z0-9_-]+=\S*")
            if pattern.match(line):
                key, val = line.split('=', 1)
                options[key] = val
        return options

    def put_porto_options(self, options_dict):
        data = ""
        for k, v in options_dict.items():
            data += k + "=" + v + '\n'

        self.put_file(StringIO(data), self._porto_options_path)

    def _format_runner_cmd(self, opts, action, agent_args):
        def params_to_string(params):
            _options = []
            for param, value in params.items():
                width = len(param) == 1 and 2 or len(param) + 2
                if value is None:
                    _opts = '{0:->{width}}'.format(param, width=width)
                else:
                    _opts = '{0:->{width}} {1}'.format(param, value, width=width)
                _options.append(_opts)
            return ' '.join(_options)

        options = params_to_string(opts)
        agent_args = params_to_string(agent_args)
        runner_command = '%s %s %s %s' % (self._runner_path, options, action, agent_args)
        logger.info("invoking agent-runner: %s", runner_command)

        return runner_command

    def start(self, **kwargs):
        # FIXME: maybe this should be moved to some preparse method
        agent_args = {}
        if kwargs.get('local_mode', False):
            kwargs.pop('local_mode')
            agent_args['l'] = ''

        if not kwargs.get('use_porto_options', False):
            kwargs.pop('use_porto_options', None)
            kwargs['ignore-porto-options'] = None

        options = {'v': 'debug'}
        options.update(kwargs)
        options['log-stdout'] = None
        options['check-alive'] = 600

        self.put_file(self.options_parser.write_options(),
                      self._vmoptions_path)

        # set running mode data
        self.run_mode = kwargs.get('mode', 'porto')
        self.agent_container = kwargs.get('container')

        self.perform('touch %s' % self.initial_startup_marker)

        # prepare cmd and execute it
        logger.info("Starting iss container...")
        cmd = self._format_runner_cmd(options, 'start', agent_args)
        result = self.perform("cd '{}' && {}".format(self.run_dir, cmd))
        logger.info("Iss container started.")

    def stop(self, **kwargs):
        """Stops agent, but does not stop running instances"""
        return self._stop_template('stop', **kwargs)

    def stop_all(self, **kwargs):
        """Stops all running instances and agent"""
        return self._stop_template('stopall', **kwargs)

    def status(self):
        """Returns status of agent"""
        cmd = self._format_runner_cmd({}, 'status', {})
        return self.perform(cmd)

    def _stop_template(self, command, **kwargs):
        pass

    def get_file_path(self, name):
        output = self.perform('find ' + self.run_dir + ' -name ' + name)
        files = output.stdout.split()
        return files

    def get_file_lines(self, file_path):
        with open(file_path, 'r') as f:
            lines = f.readlines()
        return lines

    def _get_process_pids(self, pattern):
        output = self.perform('pgrep -f "%s"' % pattern)
        return output.stdout.split()

    def get_hook_pids(self, hook_name, interpreter='bash'):
        """Return PIDs of hook processes

        Important note: This method checks only processes
        that are launched from Agent directory.
        :param hook_name: process name
        :return: list of PIDs
        """
        extended_pattern = '%s %s.*%s$' % (interpreter, self.run_dir, hook_name)
        return self._get_process_pids(extended_pattern)

    def get_child_processes_by_pid(self, parent_pid):
        """Returns PIDs of child processes

        :param parent_pid:
        :return: list of child processes PIDs
        """
        output = self.perform('pgrep -P %s' % parent_pid).stdout
        return output.split()

    def is_process_alive(self, pid):
        output = self.perform('ps -f --pid %s' % pid)
        return output.ok

    def get_property_from_manifest(self, property_name):
        jar_file = os.path.join(self.run_dir, 'iss-agent.jar')
        self.perform('jar -xf %s' % jar_file)
        output = self.perform('grep -i "%s" META-INF/MANIFEST.MF | cut -d " " -f 2' % property_name)
        return output.stdout

    def get_last_modified_timestamp(self, file_path):
        """Gets last modified timestamp of file by running stat command.

        :param file_path: path to file
        :return: timestamp (float). if timestamp can not be obtained, returns 0.0
        """
        rv = 0.0
        result = self.perform('stat -c %%Y %s' % file_path)
        if result.stdout:
            rv = float(result.stdout)

        return rv


class VMOptionsContextBuilder(object):

    def __init__(self):
        self.provider_port = None
        self.fqdn = None
        self.provider_type = "thrift"
        self.agent_ports = None
        self.certificates_dir = None
        self.agent_dir = None

    def set_provider_port(self, port):
        self.provider_port = port
        return self

    def set_fqdn(self, fqdn):
        self.fqdn = fqdn
        return self

    def set_provider_type(self, provider_type):
        self.provider_type = provider_type
        return self

    def set_agent_ports(self, agent_ports):
        self.agent_ports = agent_ports
        return self

    def set_agent_dir(self, agent_dir):
        self.agent_dir = agent_dir
        return self

    def set_certificates_dir(self, certificates_dir):
        self.certificates_dir = certificates_dir
        return self

    def build(self):
        for field, value in self.__dict__.items():
            if value is None:
                raise ValueError("Attibute {attr} is not set".format(attr=field))
        return VMOptionsContext(self.provider_port, self.fqdn, self.provider_type,
                                self.agent_ports, self.certificates_dir,
                                self.agent_dir)


class VMOptionsContext(object):

    def __init__(self, provider_port, fqdn, provider_type, agent_ports, certificates_dir, agent_dir):
        self.provider_port = provider_port
        self.fqdn = fqdn
        self.provider_type = provider_type
        self.agent_ports = agent_ports
        self.certificates_dir = certificates_dir
        self.agent_dir = agent_dir


def common_cacher_agent_vmoptions(context):
    return common_agent_options(context) + [
        "-Dagent.hostConfiguration.primaryProvider.host={}".format(context.fqdn),
        "-Dagent.hostConfiguration.primaryProvider.port={}".format(context.provider_port),
        "-Dagent.hostConfiguration.primaryProvider.type={}".format(context.provider_type),
        "-Dagent.shardtracker.host={}".format(context.fqdn),
        "-Dagent.shardtracker.timeout=3000ms",
        "-Dagent.feedback.resendPeriod=1s",
    ]


def common_agent_options(context):
    return [
        "-Dagent.webapp.port={}".format(context.agent_ports['webapp_port']),
        "-Dagent.jmx.mp.port={}".format(context.agent_ports['jmx_port']),
        "-Dagent.spaceToLeaveOnDisk=3g",
        "-Dagent.fetcher.allow_files_fetch_task=true",
        "-Dagent.shards.downloadAttempts=1",
        "-Dagent.shards.downloadFrequencyLimit=5s",
        "-agentlib:jdwp=transport=dt_socket,server=y,suspend=n,address=localhost:{}".format(
            context.agent_ports['agent_debug_port']),
        "-Dagent.secure_webapp.port={}".format(context.agent_ports['secure_webapp_port']),
        "-Dagent.state.path_to_initial_startup_marker={}".format(
            os.path.join(context.agent_dir, '_initial_start_marker')),
    ]
