from iss.common.agent_api_client import TCPAgentRestAPIClient
from iss.common.issagent import IssAgent
from iss.common.vmoptions_parser import VMOptionsParser

from hashlib import md5
import subprocess
import os

try:
    xrange
except NameError:
    xrange = range


def get_next(port_generator):
    try:
        return next(port_generator)
    except Exception as e:
        raise Exception("not enough ports in pool", e)


class IssAgentPorts(object):
    def __init__(self, port_generator):
        self.webapp = get_next(port_generator)
        self.secure_webapp = get_next(port_generator)
        self.jmx = get_next(port_generator)
        self.debug = get_next(port_generator)
        self.yp_notify = get_next(port_generator)


class IssAgentInstance(object):
    def __init__(self, client, api_client, hostname, allocatable_subnet):
        self.client = client
        self.api_client = api_client
        self.hostname = hostname
        self.allocatable_subnet = allocatable_subnet

    def start(self, **kwargs):
        self.client.start(**kwargs)

    def stop(self, **kwargs):
        self.client.stop(**kwargs)


class IssLocal(object):
    ISS_TEMPLATES_DIR = '/db/iss3'
    ISS_ROOT_ON_STORAGE = 'db/iss3'

    def __init__(self, agent_id, storage_dir, host_fqdn, allocatable_subnet, yp_agent_host, yp_agent_port, port_generator):
        self.agent_id = agent_id
        self.host_fqdn = host_fqdn
        self.allocatable_subnet = allocatable_subnet
        self.yp_agent_host = yp_agent_host
        self.yp_agent_port = yp_agent_port
        self.port_generator = iter(port_generator)
        self.agent_storage = storage_dir
        self.agent_test_dir = os.path.join(storage_dir, self.ISS_ROOT_ON_STORAGE)

    def start(self):
        agent_ports = IssAgentPorts(self.port_generator)

        agent_object = IssAgent(
            self.agent_test_dir,
            self.ISS_TEMPLATES_DIR,
            self.agent_test_dir
        )

        self.prepare_env(agent_object)

        agent_object.install()

        self.agent = agent_object

        vmoptions = self._vmoptions(self.agent_test_dir, agent_ports)

        agent_object.configure([
            (vmoptions, IssAgent.ISS_AGENT_VMOPTIONS)
        ])

        webapp_log_dir = \
            os.path.join(self.agent_test_dir, 'webapp_logs')

        if not os.path.exists(webapp_log_dir):
            os.makedirs(webapp_log_dir)

        return IssAgentInstance(
            agent_object,
            TCPAgentRestAPIClient(
                "localhost",
                agent_ports.webapp,
                webapp_log_dir
            ),
            self.host_fqdn,
            self.allocatable_subnet
        )

    def prepare_env(self, agent_object):
        agent_object.perform("rm /etc/cron.d/yandex-iss-agent")
        agent_object.perform("portoctl destroy iss-agent")
        agent_object.perform("touch /db/iss3/_initial_start_marker")

    def stop(self):
        self.agent.perform("cp /var/log/portod.log %s" % self.agent_test_dir)
        self.agent.perform("portoctl destroy iss-agent")

    def _vmoptions(self, agent_dir, agent_ports):
        vmoptions_path = os.path.join('/db/iss3/iss-agent.vmoptions')

        with open(vmoptions_path, 'r') as f:
            vmoptions_content = f.readlines()

        initial_marker = os.path.join(agent_dir, '_initial_start_marker')

        agent_id = md5(agent_dir.encode('utf-8')).hexdigest()

        parser = VMOptionsParser(vmoptions_content)
        parser.update_options_list([
            "-agentlib:jdwp=transport=dt_socket,server=y,suspend=n,address=localhost:{}".format(
                agent_ports.debug),
            "-Dagent.webapp.port={}".format(agent_ports.webapp),
            "-Dagent.jmx.mp.port={}".format(agent_ports.jmx),
            "-Dagent.spaceToLeaveOnDisk=-1",
            "-Dagent.shards.downloadAttempts=1",
            "-Dagent.shards.downloadFrequencyLimit=5s",
            "-Dagent.secure_webapp.port={}".format(agent_ports.secure_webapp),
            "-Dagent.hostConfiguration.providersNumber=0",
            "-Dagent.hostConfiguration.useDefaultServerInfo=true",
            "-Dagent.hostConfiguration.ypProvider.enabled=true",
            "-Dagent.hostConfiguration.ypProvider.allowApply=true",
            "-Dagent.hostConfiguration.ypProvider.heartbeatInterval=2s",
            "-Dagent.hostConfiguration.ypProvider.host={}".format(self.yp_agent_host),
            "-Dagent.hostConfiguration.ypProvider.port={}".format(self.yp_agent_port),
            "-Dagent.hostConfiguration.ypProvider.notify.port={}".format(
                agent_ports.yp_notify),
            "-Dagent.porto.metaProcessContainer={}".format('ISS'),
            "-Dagent.fetcher.allow_files_fetch_task=true",
            "-Dagent.fetcher.containerized.meta_name={}".format(agent_id[:4]),
            "-Dagent.state.path_to_initial_startup_marker={}".format(
                initial_marker),
            "-Dagent.state.stateStoragePath={}".format(agent_dir),
            "-Dagent.rootOnStorage={}".format(self.ISS_ROOT_ON_STORAGE),
            "-Dagent.defaultStorage=%s" % self.agent_storage,
            "-Dagent.storageMountPoints.0=%s" % self.agent_storage,
            "-Dagent.storageMountPoints.1=/ssd",
            "-Dagent.sync.autoSyncPeriod=2s",
            "-Dagent.sync.autoUpdatePeriod=5s",
            "-Dagent.loglevel=DEBUG",
        ])
        return parser.write_options()
