from iss.common.issagent import HookNames

"""A collection of most frequently used conditions.

Each condition should be a callable that in general should not expect
to receive any arguments during the call. This could be done by implementing
conditions as classes with __call__ method or by using function closures.

Expected conditions are not of much use in their own, but are powerful
when used in wait_for_assertion or assertion_holds functions.
"""


def assertion(value):
    assert value
    return value


def negative_assertion(value):
    return assertion(not value)


def agent_stopped(agent):
    """An expectation for checking is agent stopped"""

    def inner():
        assert not agent.get_agent_pid()

    return inner


def _start_hook_pids(agent, check_function, interpreter='bash'):
    """An expectation regarding active instance pids"""

    def inner():
        return check_function(agent.get_hook_pids(HookNames.START_HOOK, interpreter=interpreter))

    return inner


def any_instance_running(agent, interpreter='bash'):
    return _start_hook_pids(agent, assertion, interpreter)


def no_instances_running(agent, interpreter='bash'):
    return _start_hook_pids(agent, negative_assertion, interpreter)


def are_same_instances_running(agent, pids):
    """An expectation for checking that exactly same instances are still running

    If processes with given PIDs are running, but there are some other processes
    running also, then this situation will be considered as failure of condition.
    """

    def inner():
        expected_pids = pids
        if isinstance(expected_pids, basestring):
            expected_pids = [expected_pids]
        new_pids = agent.get_hook_pids(HookNames.START_HOOK)
        return assertion(new_pids == expected_pids)

    return inner


def instance_restarted(agent, instance_pid):
    """An expectation for checking is instance restarted

    This is done by checking that there is no process with given PID alive
    and ensuring that agent has EXACTLY ONE other active instance running.
    Thus this expected condition is not usable with multiple running instances.
    """

    def inner():
        pids = agent.get_hook_pids(HookNames.START_HOOK)
        assert len(pids) == 1
        return assertion(instance_pid != pids[0])

    return inner


def mtime_not_changed(agent, file_path, control_timestamp):
    """An expectation for checking that mtime of given file equals to control timestamp"""

    def inner():
        timestamp = agent.get_last_modified_timestamp(file_path)
        assert timestamp and timestamp == control_timestamp

    return inner


def mtime_changed(agent, file_path, control_timestamp):
    """An expectation for checking that mtime of given file is not equal to control timestamp"""

    def inner():
        timestamp = agent.get_last_modified_timestamp(file_path)
        assert timestamp and timestamp != control_timestamp

    return inner


def wait_for_container_state(porto_api, container_re, expected_state='running'):
    def inner():
        containers = porto_api.get_containers_by_regex(container_re)
        assert containers and containers[0].GetData('state') == expected_state
        return containers[0]

    return inner
