import time

__author__ = 'sashakruglov@yandex-team.ru'

DEFAULT_AWAIT_TIMEOUT = 90
DEFAULT_HOLD_TIMEOUT = 30
DEFAULT_POLL_FREQUENCY = 0.5


def _exception_set(exc_type):
    from collections import Iterable

    exc_types = exc_type
    if not isinstance(exc_type, Iterable):
        exc_types = [exc_types]

    exc_types = set(exc_types)
    exc_types.add(AssertionError)

    return exc_types


def wait_for_assertion(assertion, timeout=DEFAULT_AWAIT_TIMEOUT, poll_frequency=DEFAULT_POLL_FREQUENCY,
                       exc_type=Exception):
    """Waits till given assertion will passes

    :param assertion: callable
    :param timeout: time to wait until condition returns True
    is not met before timeout
    :param poll_frequency: how often should condition be checked
    :param exc_type: type (or iterable of types) of exception to handle (by default it's AssertionError)
    :return: value produced by condition object
    :raises: TimeoutException if assertion is not met before timeout
    :raises: AssertionError if assertion never passed
    """

    exc_types = _exception_set(exc_type)

    start = time.time()
    end = start + timeout
    last_exception = None
    while end > time.time():
        try:
            return assertion()
        except BaseException as e:
            if not any([isinstance(e, _) for _ in exc_types]):
                raise e
            last_exception = e
        time.sleep(poll_frequency)
    raise last_exception


def negate_assertion(assertion, exc_type=AssertionError):
    exc_types = _exception_set(exc_type)

    class PassedException(Exception):
        pass

    def negative_assertion(*args, **kwargs):
        try:
            assertion(*args, **kwargs)
            raise PassedException
        except PassedException:
            raise AssertionError("Assertion failure expected")
        except BaseException as e:
            if not any([isinstance(e, _) for _ in exc_types]):
                raise e

    return negative_assertion


def wait_for_assertion_failure(assertion, timeout=DEFAULT_AWAIT_TIMEOUT, poll_frequency=DEFAULT_POLL_FREQUENCY,
                               exc_type=AssertionError):
    """Waits till given assertion will fail

    :param assertion: callable
    :param timeout: time to wait until condition returns True
    is not met before timeout
    :param poll_frequency: how often should condition be checked
    :param exc_type: type (or iterable of types) of exception to handle (by default it's AssertionError)
    :return: value produced by condition object
    :raises: TimeoutException if assertion is not met before timeout
    :raises: AssertionError if assertion never passed
    """

    wait_for_assertion(negate_assertion(assertion), timeout=timeout, poll_frequency=poll_frequency, exc_type=exc_type)


def assertion_holds(assertion, time_to_check=DEFAULT_HOLD_TIMEOUT, poll_frequency=DEFAULT_POLL_FREQUENCY):
    """Checks that given assertion holds for specified timeout

    :param assertion: callable which might throw AssertionError
    :param time_to_check: time to check that condition is still True
    :param poll_frequency: how often should condition be checked
    :raises: AssertionError if assertion does not pass for given time frame
    """
    start = time.time()
    end = start + time_to_check
    while end > time.time():
        try:
            assertion()
        except AssertionError as e:
            raise e
        time.sleep(poll_frequency)
