import functools
import pwd
import socket
from contextlib import closing

import fnmatch
import os
import psutil
import re
import sys

try:
    xrange
except NameError:
    xrange = range

__author__ = 'sashakruglov@yandex-team.ru'


def locate(pattern, root=os.curdir):
    """
    Locate all files matching supplied filename pattern in and below
        supplied root directory.
    """
    located_files = []
    for path, dirs, files in os.walk(os.path.abspath(root)):
        for filename in fnmatch.filter(files, pattern):
            located_files.append(os.path.join(path, filename))
    return located_files


def dict_to_thrift_map(d, parent_key='', result=None):
    """Makes dictionary suitable to use as thrift map object

    :param d: dictionary
    :param parent_key: used internally
    :param result: used internally
    :return: modified dictionary with flatten keys
    """
    result = result or {}
    fmt = parent_key and '%s/%s' or '%s%s'
    for k, v in d.items():
        extended_key = fmt % (parent_key, k)

        if isinstance(v, list):
            v = {str(i): v for i, v in enumerate(v)}

        if isinstance(v, dict):
            dict_to_thrift_map(v, extended_key, result)
        else:
            # properties should be strings
            result[extended_key] = str(v)
    return result


def get_processes_number(process_name_pattern):
    """
    Get the number of processes which names match the pattern.
    """
    login = pwd.getpwuid(os.getuid())[0]

    number_of_processes = 0

    for process in psutil.get_process_list():
        try:
            cmdline = process.cmdline
            for line in cmdline:
                if process.username == login and re.match(process_name_pattern, line):
                    number_of_processes += 1
                    break
        except psutil.NoSuchProcess:
            continue

    return number_of_processes


def ensure_uniqueness(func):
    ports = [1023]

    @functools.wraps(func)
    def wrapper():
        result = func(ports[-1])
        ports.append(result)
        return result

    return wrapper


@ensure_uniqueness
def get_free_port(start):
    for port in get_free_port_in_range(start + 1, 8092):
        return port


def get_free_port_in_range(start, end):
    sys.stderr.write("GetPort: check in range %d - %d" % (start, end))
    for i in xrange(start, end):
        sock = socket.socket(socket.AF_INET6, socket.SOCK_STREAM)
        sock.setsockopt(socket.SOL_SOCKET, socket.SO_REUSEADDR, 1)

        try:
            sock.bind(('', i))
        except socket.error:
            continue

        port = sock.getsockname()[1]
        sock.close()
        sys.stderr.write("GetPort: got free port %d\n" % i)
        yield port


def share_source(source, dir_name, file_name):
    full_path_name = os.path.join(dir_name, file_name)
    with open(full_path_name, 'w') as f:
        f.write(source)
    return "file://{}".format(full_path_name)


def strip_color_codes(text):
    ansi_escape = re.compile(r'\x1b[^m]*m')
    return ansi_escape.sub('', text)


def strip_new_line(text):
    return text.strip('\r').strip('\n')


def strip_junk(text, junk_end):
    result = text.split(junk_end)
    if len(result) == 2:
        return result[1]
    else:
        return text


def strip_margin(string, char='\|'):
    """
    Strips string with specific margin character.
    Inspired by Scala.

    >>> strip_margin('''
    ...    |just
    ...    | a
    ...    ||test
    ...    ''')
    'just\\n a\\n|test'

    :param string:
    :param char:
    """
    import re
    compiled = re.compile(r'^(\s*)%s(.*)$' % char)
    result = []

    lines = string.splitlines()
    length = len(lines)
    for i, line in enumerate(lines):
        stripped = line.strip()
        if (i == 0 or i == length - 1) and not stripped:
            continue
        if line.strip().startswith('|'):
            result.append(compiled.sub(r'\2', line))
        else:
            result.append(line)
    return "\n".join(result)


def is_socket_opened(socket_to_test):
    with closing(socket.socket(socket.AF_INET6, socket.SOCK_STREAM)) as sock:
        sock.settimeout(2)
        return sock.connect_ex((socket_to_test[0], socket_to_test[1])) == 0


def touch(path_to_file):
    with open(path_to_file, 'a'):
        os.utime(path_to_file, None)
