#include <linux/module.h>

#include <net/net_namespace.h>
#include <net/tcp.h>

MODULE_AUTHOR("Dmitry Yakunin <zeil@yandex-team.ru>");
MODULE_DESCRIPTION("Fix parameters in all net namespaces");
MODULE_LICENSE("GPL");

extern struct rw_semaphore net_rwsem;

static int tcp_syn_retries = 2;
static int tcp_syn_retries_set(const char *val, const struct kernel_param *kp)
{
    struct net *net;
    int n = 0, ret;

    ret = kstrtoint(val, 10, &n);
    if (ret || n < 1 || n > MAX_TCP_SYNCNT)
        return -EINVAL;

    tcp_syn_retries = n;

    down_read(&net_rwsem);
    for_each_net(net) {
        net->ipv4.sysctl_tcp_syn_retries = tcp_syn_retries;
    }
    up_read(&net_rwsem);

    return 0;
}
static const struct kernel_param_ops tcp_syn_retries_ops = {
    .set = tcp_syn_retries_set,
    .get = param_get_int,
};
module_param_cb(tcp_syn_retries, &tcp_syn_retries_ops, &tcp_syn_retries, 0664);
MODULE_PARM_DESC(tcp_syn_retries, "Sysctl net.ipv4.tcp_syn_retries");

static int tcp_synack_retries = 2;
static int tcp_synack_retries_set(const char *val, const struct kernel_param *kp)
{
    struct net *net;
    int ret;

    ret = param_set_int(val, kp);
    if (!ret) {
        down_read(&net_rwsem);
        for_each_net(net) {
            net->ipv4.sysctl_tcp_synack_retries = tcp_synack_retries;
        }
        up_read(&net_rwsem);
    }

    return ret;
}
static const struct kernel_param_ops tcp_synack_retries_ops = {
    .set = tcp_synack_retries_set,
    .get = param_get_int,
};
module_param_cb(tcp_synack_retries, &tcp_synack_retries_ops, &tcp_synack_retries, 0664);
MODULE_PARM_DESC(tcp_synack_retries, "Sysctl net.ipv4.tcp_synack_retries");

static int __init netns_fixer_init(void)
{
    return 0;
}

static void __exit netns_fixer_exit(void)
{
}

module_init(netns_fixer_init);
module_exit(netns_fixer_exit);
