#!/usr/bin/env python3

import subprocess
import contextlib
import tempfile
import argparse
import shutil
import sys
import os


@contextlib.contextmanager
def tmpdir(**kwargs):
    tmp = tempfile.mkdtemp(**kwargs)
    try:
        yield tmp
    finally:
        shutil.rmtree(tmp)


BUILD_DEPS = [
    'build-essential',
    'zlib1g-dev',
    'binutils-dev',
    'libelf-dev',
    'libiberty-dev',
]

def run(args, **kwargs):
    print("+ '" + "' '".join(args) + "'", flush=True)
    return subprocess.check_call(args, **kwargs)


def install_deps(args):
    cmd = ['sudo', 'apt-get', 'install', '--no-install-recommends', '-y'] + BUILD_DEPS
    if args.dry_run:
        print("# Install build dependencies.")
        print(" ".join(cmd))
        sys.exit(0)
    else:
        run(['sudo', 'apt-get', 'update', '-y'])
        run(cmd)


def build_bin(args):
    if args.install_deps:
        install_deps(args)

    with tmpdir(prefix='build-' + os.path.basename(args.src)) as build_dir:
        run(['tar', '-xf', os.path.abspath(args.src), '--strip-components=1'], cwd=build_dir)
        tool_dir = build_dir + '/tools/bpf/bpftool'
        run(['sed', '-i', '/CFLAGS += -O2/a CFLAGS += -static', 'Makefile'], cwd=tool_dir)
        run(['sed', '-i', 's/LIBS = $(LIBBPF) -lelf -lz/LIBS = $(LIBBPF) -lelf -lbfd -lopcodes -lz -ldl -liberty/g', 'Makefile'], cwd=tool_dir)
        env = os.environ.copy()
        env['VMLINUX_BTF']=''
        run(['make', '-j', str(os.sysconf('SC_NPROCESSORS_ONLN'))], cwd=tool_dir, env=env)
        run(['tar', '-caf', os.path.abspath(args.out), 'bpftool', '--transform', 's,^,bpftool/,'], cwd=tool_dir)

if __name__ == '__main__':
    parser = argparse.ArgumentParser()
    subparsers = parser.add_subparsers(title="Possible extra actions", dest='command')
    deps = subparsers.add_parser(name="install-deps")
    deps.set_defaults(handle=install_deps)
    deps.add_argument("--dry-run", default=False, action='store_true')
    build = subparsers.add_parser(name="build")
    build.set_defaults(handle=build_bin)
    build.add_argument("--dry-run", default=False, action='store_true')
    build.add_argument("--install-deps", default=False, action='store_true')
    build.add_argument("src", help="source tarball")
    build.add_argument("out", help="output tarball")
    args = parser.parse_args()
    args.handle(args)
