#!/usr/bin/env python3

import subprocess
import contextlib
import tempfile
import argparse
import shutil
import sys
import os


@contextlib.contextmanager
def tmpdir(**kwargs):
    tmp = tempfile.mkdtemp(**kwargs)
    try:
        yield tmp
    finally:
        shutil.rmtree(tmp)


BUILD_DEPS = [
    'docker.io',
]

def run(args, **kwargs):
    print("+ '" + "' '".join(args) + "'", flush=True)
    return subprocess.check_call(args, **kwargs)


def install_deps(args):
    cmd = ['sudo', 'apt-get', 'install', '--no-install-recommends', '-y'] + BUILD_DEPS
    if args.dry_run:
        print("# Install build dependencies.")
        print(" ".join(cmd))
        sys.exit(0)
    else:
        run(['sudo', 'apt-get', 'update', '-y'])
        run(cmd)


def build_bin(args):
    if args.install_deps:
        install_deps(args)

    with tmpdir(prefix='build-' + os.path.basename(args.src)) as build_dir:
        run(['tar', '--strip-components=1', '-xf', os.path.abspath(args.src)], cwd=build_dir)
        run(['sed', '-i', '1inameserver 2a02:6b8:0:3400::5005', '/etc/resolv.conf'])
        run(['bash', '-c', 'echo \'{"dns":["2a02:6b8:0:3400::5005"]}\' > /etc/docker/daemon.json'])
        run(['service', 'docker', 'restart'])
        run(['sed', '-i', '/WORKDIR \//a RUN wget http://dl-cdn.alpinelinux.org/alpine/v3.12/main/x86_64/linux-headers-5.4.5-r1.apk\\nRUN apk add ./linux-headers-5.4.5-r1.apk', 'Dockerfile.alpine'], cwd=build_dir + '/docker')
        run(['bash', '-c', 'docker build --network host -t bpftrace-builder-alpine -f Dockerfile.alpine .'], cwd=build_dir + '/docker')
        run(['bash', '-c', 'docker run --network host -v $(pwd):$(pwd) -e STATIC_LINKING=ON -e STATIC_LIBC=ON -e RUN_TESTS=0 bpftrace-builder-alpine $(pwd)/build-release Release -j$(nproc)'], cwd=build_dir)
        run(['tar', '-caf', os.path.abspath(args.out), '-C', 'tools', '.', '-C', '../build-release/src', 'bpftrace', '--transform', 's,^,bpftrace/,'], cwd=build_dir)

if __name__ == '__main__':
    parser = argparse.ArgumentParser()
    subparsers = parser.add_subparsers(title="Possible extra actions", dest='command')
    deps = subparsers.add_parser(name="install-deps")
    deps.set_defaults(handle=install_deps)
    deps.add_argument("--dry-run", default=False, action='store_true')
    build = subparsers.add_parser(name="build")
    build.set_defaults(handle=build_bin)
    build.add_argument("--dry-run", default=False, action='store_true')
    build.add_argument("--install-deps", default=False, action='store_true')
    build.add_argument("src", help="source tarball")
    build.add_argument("out", help="output tarball")
    args = parser.parse_args()
    args.handle(args)
