#!/usr/bin/env python3

import os
import argparse
import tempfile
import subprocess


def run(args, **kwargs):
    print("+ '" + "' '".join(args) + "'")
    return subprocess.check_call(args, **kwargs)


def build_tarball(args):
    make_args = ['make', '-j' + str(args.jobs), '-C', args.sources]

    if not args.verbose:
        make_args += ['-s']

    if args.build_dir is not None:
        make_args += ['O=' + args.build_dir]
    else:
        args.build_dir = args.sources

    if args.config is not None:
        make_args += ['KCONFIG_CONFIG=' + args.config]

    if args.arch is not None:
        make_args += ['ARCH=' + args.arch]

    if args.local_version is not None:
        make_args += ['LOCALVERSION=' + args.local_version]

    # TODO: support llvm toolchain, right now it's corectly works only with gcc
    if args.with_ccache:
        make_args += ['KBUILD_BUILD_TIMESTAMP=""']
        make_args += ['CC=ccache ' + os.environ.get('CROSS_COMPILE', '') + 'gcc']

    run(make_args + ['oldconfig'])
    release = subprocess.check_output(make_args + ['-s', 'kernelrelease']).decode('utf8').strip()
    print("kernel release:", release)
    run(make_args)

    if args.tarball is None:
        args.tarball = 'linux-' + release + '.tar.gz'

    with tempfile.TemporaryDirectory(dir=os.path.abspath(args.build_dir), prefix='install-') as install_dir:
        os.chmod(install_dir, 0o775)
        os.mkdir(install_dir + '/boot', mode=0o775)
        install_args = ['INSTALL_PATH=' + install_dir + '/boot',
                        'INSTALL_MOD_STRIP=1',
                        'INSTALL_MOD_PATH=' + install_dir]
        if args.install_firmware:
            install_args += ['INSTALL_FW_PATH=' + install_dir + 'lib/firmware/' + release]
        run(make_args + install_args +  ['modules_install', 'install'])
        if args.install_vmlinux:
            run(['cp', args.build_dir + '/vmlinux', install_dir + '/boot/vmlinux-' + release])
        if args.install_debug:
            os.makedirs(install_dir + '/usr/lib/debug/boot', mode=0o775, exist_ok=True)
            run(['cp', args.build_dir + '/vmlinux', install_dir + '/usr/lib/debug/boot/vmlinux-' + release])
            run(make_args + ['INSTALL_MOD_PATH=' + install_dir + '/usr/lib/debug', 'modules_install'])
        run(['tar', '-caf', os.path.abspath(args.tarball), '.'], cwd=install_dir)


def main():
    parser = argparse.ArgumentParser()

    subparsers = parser.add_subparsers(title="Possible extra actions", dest='command')
    subparsers.required = True

    tarball = subparsers.add_parser(name="tarball")
    tarball.set_defaults(handle=build_tarball)
    tarball.add_argument('-j', '--jobs', type=int, default=os.sysconf('SC_NPROCESSORS_ONLN'), help='default: %(default)s')
    tarball.add_argument('--verbose', action='store_true')
    tarball.add_argument('--build-dir', help='default: build in source dir')
    tarball.add_argument('--arch')
    tarball.add_argument('--config', help="default: .config")
    tarball.add_argument('--local-version')
    tarball.add_argument('--with-ccache', action='store_true')
    tarball.add_argument('--install-headers', action='store_true')
    tarball.add_argument('--install-firmware', action='store_true')
    tarball.add_argument('--install-vmlinux', action='store_true')
    tarball.add_argument('--install-debug', action='store_true')
    tarball.add_argument("sources")
    tarball.add_argument("tarball", nargs='?', help='default: linux-<release>.tar.gz')
    tarball.add_argument("debuginfo", nargs='?', help='default: linux-<release>-debuginfo.tar.gz')

    args = parser.parse_args()
    args.handle(args)


if __name__ == "__main__":
    main()
