#!/usr/bin/env python3

import os
import re
import sys
import json
import socket
import codecs
import argparse
import datetime
import requests
import subprocess


WALLE_API = 'https://api.wall-e.yandex-team.ru'

# Direct access instead balancer coroner.n.yandex-team.ru
# CORONER_API = 'http://coroner-vla-1.vla.yp-c.yandex.net'
CORONER_API = 'http://coroner.n.yandex-team.ru'

CORONER_LOGS = '/logs/netconsole'

DEFAULT_DOMAIN = 'search.yandex.net'


def pretty_json(obj):
    return json.dumps(obj, default=lambda obj: obj.__dict__,
                      sort_keys=True, indent=2, separators=(',', ': '))


def print_json(obj, out=sys.stdout):
    json.dump(obj, out,
              default=lambda obj: obj.__dict__,
              sort_keys=True,
              indent=2,
              separators=(',', ': '))


def walle_hosts(**params):
    params.setdefault('offset', 0)
    params.setdefault('limit', 10000)
    ret = []
    while True:
        result = requests.get(WALLE_API + '/v1/hosts', params=params).json()['result']
        nr = len(result)
        ret.extend(result)
        if nr < params['limit']:
            break
        params['offset'] += nr
    return ret


class EventParser(object):
    __patterns = [
'(?P<error_mce_panic>Kernel panic - not syncing: Fatal [Mm]achine check)',     # noqa: F122
'(?P<error_mce_panic_timeout>Kernel panic - not syncing: Timeout synchronizing machine check over CPUs)',
'(?P<error_mce_hardware>mce: \[Hardware Error\].*)',
'(?P<error_kill_idle>Kernel panic - not syncing: Attempted to kill the idle task!)',
'(?P<error_kill_init>Kernel panic - not syncing: Attempted to kill init!.*)',
'(?P<error_double_fault>PANIC: double fault.*)',

'(?P<error_fault_null>BUG: unable to handle kernel NULL pointer dereference at .*)',
'(?P<error_fault>BUG: unable to handle kernel paging request at .*)',
'(?P<error_sleep_in_atomic>BUG: sleeping function called from invalid context at .*)',
'(?P<error_sched_in_atomic>BUG: scheduling while atomic: .*)',
'(?P<error_kernel>kernel BUG at .*)',
'(?P<error_gpf>general protection fault: .*)',
'(?P<error_corrupted_page_table>.*Corrupted page table.*)',
'(?P<error_bad_page_map>BUG: Bad page map in process.*)',
'(?P<error_bad_rss_counter>BUG: Bad rss-counter state.*)',
'(?P<error_divide_error>divide error: .*)',

'(?P<cont_invalid_opcode>invalid opcode: .*)',

'(?P<warn_ext4_error>EXT4-fs error.*)',
'(?P<warn_ext4_warning>EXT4-fs warning.*)',
'(?P<warn_ext4_delalloc_fail>EXT4-fs \(.*\): Delayed block allocation failed.*)',
'(?P<warn_quota_error>Quota error \(device .*)',

'(?P<warn_block_eio>blk_update_request: I/O error.*)',
'(?P<warn_scsi_ure>sd.*Sense: Unrecovered read error.*)',
'(?P<warn_scsi_error>sd.*FAILED Result: hostbyte=.*)',

'(?P<warn_nvme_reset>nvme .*: Failed status:.* reset controller)',
'(?P<warn_nvme_eio>nvme .*: Cancelling I/O [0-9]+ QID [0-9]+)',

'(?P<warn_page_dump>page dumped because: .*)',

'(?P<warn_unregister_netdevice>unregister_netdevice: waiting for .*)',

'(?P<info_acpi_error>ACPI Error: .*)',

'(?P<info_page_alloc_fail>.*page allocation failure(?:: order:(?P<dump_page_order>[0-9]+), mode:(?P<dump_page_gfp>0x[0-9]+))?.*)',
'(?P<info_slub_alloc_fail>SLUB: Unable to allocate memory on node .* \(gfp=(?P<dump_slub_gfp>0x[0-9a-f]+)\))',

'(?P<info_oom>.*invoked oom-killer(?:: gfp_mask=(?P<dump_oom_gfp>0x[0-9a-f]+), \
order=(?P<dump_oom_order>[0-9]+), oom_score_adj=(?P<dump_oom_score_adj>-?[0-9]+))?.*)',


'(?P<info_mem>Mem-Info:.*)',
'active_anon:(?P<dump_mem_active_anon>[0-9]+) inactive_anon:(?P<dump_mem_inactive_anon>[0-9]+) isolated_anon:(?P<dump_mem_isolated_anon>[0-9]+)',
' active_file:(?P<dump_mem_active_file>[0-9]+) inactive_file:(?P<dump_mem_inactive_file>[0-9]+) isolated_file:(?P<dump_mem_isolated_file>[0-9]+)',
' unevictable:(?P<dump_mem_unevictable>[0-9]+) dirty:(?P<dump_mem_dirty>[0-9]+) writeback:(?P<dump_mem_writeback>[0-9]+) unstable:(?P<dump_mem_unstable>[0-9]+)',
' slab_reclaimable:(?P<dump_mem_slab_reclaimable>[0-9]+) slab_unreclaimable:(?P<dump_mem_slab_unreclaimable>[0-9]+)',
' mapped:(?P<dump_mem_mapped>[0-9]+) shmem:(?P<dump_mem_shmem>[0-9]+) pagetables:(?P<dump_mem_pagetables>[0-9]+) bounce:(?P<dump_mem_bounce>[0-9]+)',
' free:(?P<dump_mem_free>[0-9]+) free_pcp:(?P<dump_mem_free_pcp>[0-9]+) free_cma:(?P<dump_mem_free_cma>[0-9]+)',
'(?P<dump_mem_node>Node [0-9]+ .*)',
'(?P<dump_mem_pagecache>[0-9]+) total pagecache pages',
'(?P<dump_mem_swap_cache>[0-9]+) pages in swap cache',
'Free swap  = (?P<dump_mem_free_swap_kb>[0-9]+)kB',
'Total swap = (?P<dump_mem_total_swap_kb>[0-9]+)kB',
'(?P<dump_mem_pages_ram>[0-9]+) pages RAM',
'(?P<dump_mem_higmem_movable_only>[0-9]+) pages HighMem/MovableOnly',
'(?P<dump_mem_pages_reserved>[0-9]+) pages reserved',
'(?P<dump_mem_pages_hwpoisoned>[0-9]+) pages hwpoisoned',

'Task in (?P<dump_oom_memcg_victim>.*) killed as a result of limit of (?P<dump_oom_memcg_causer>.*)',
'memory: usage (?P<dump_oom_memcg_memory_usage_kb>[0-9]+)kB, limit (?P<dump_oom_memcg_memory_limit_kb>[0-9]+)kB, failcnt (?P<dump_oom_memcg_memory_failcnt>[0-9]+)',
'kmem: usage (?P<dump_oom_memcg_kmem_usage_kb>[0-9])+kB, limit (?P<dump_oom_memcg_kmem_limit_kb>[0-9]+)kB, failcnt (?P<dump_oom_memcg_kmem_failcnt>[0-9]+)',
'anon: usage (?P<dump_oom_memcg_anon_usage_kb>[0-9])+kB, limit (?P<dump_oom_memcg_anon_limit_kb>[0-9]+)kB, failcnt (?P<dump_oom_memcg_anon_failcnt>[0-9]+)',
'Memory cgroup stats for (?P<dump_oom_memcg_stats>.*)',

'(?P<info_oom_kill>Killed process (?P<dump_oom_kill_pid>[0-9]+) \((?P<dump_oom_kill_comm>.*)\) (?P<dump_oom_kill_stats>.*))',

'(?P<info_nmi_backtrace>NMI backtrace for cpu \d+)',

'(?:NMI watchdog: )?BUG: (?P<warn_soft_lockup>soft lockup - .*)',
'(?:NMI watchdog: )?BUG: (?P<warn_hard_lockup>hard lockup - .*)',
'(?:NMI watchdog: )?Watchdog detected (?P<warn_hard_LOCKUP>hard LOCKUP .*)',
'BUG: (?P<warn_workqueue_lockup>workqueue lockup .*)',

# from kernel/hung_task.c:130
'INFO: (?P<info_task_lockup>task .* blocked for more than .* seconds.)',
'      (?:Not tainted|Tainted: (?P<dump_taint_hung_task>[A-Z ]+)) (?P<dump_release_hung_task>.*) (?P<dump_version_hung_task>.*)',
'INFO: (?P<info_rcu_lockup>.* detected stalls on CPUs/tasks.*)',
'INFO: (?P<info_rcu_self_lockup>.* self-detected stall on CPU.*)',
'INFO: (?P<info_rcu_task_lockup>rcu_tasks detected stalls on tasks.*)',
'(?P<info_perf_lockup>perf interrupt took too long .*)',
'.*: (?P<info_page_alloc_stall>page alloc(?:a)?tion stalls for .*)',

'microcode: (?P<info_microcode_early_update>microcode updated early .*)',

'(?P<info_boot>Linux version (?P<dump_boot_release>[^ ]*) (?P<dump_boot_build>.*))',
'Command line: (?P<dump_boot_cmdline>.*)',

'\[ (?P<error_lockdep>BUG: lock held when returning to user space!|\
BUG: Nested lock was not taken|\
BUG: bad unlock balance detected!|\
BUG: bad contention detected!|\
BUG: held lock freed!|\
BUG: .* still has locks held!|\
INFO: suspicious RCU usage.|\
INFO: possible circular locking dependency detected|\
INFO: .*unsafe lock order detected|\
INFO: possible recursive locking detected|\
INFO: inconsistent lock state|\
INFO: possible irq lock inversion dependency detected).*',

'(?P<error_panic>PANIC: .*)',
'(?P<error_bug>BUG.*)',
'(?P<warn_warnings>WARNING: .*)',

'Modules linked in: (?P<dump_modules>.*)',
'CPU: (?P<dump_cpu>\d+) PID: (?P<dump_pid>\d+) Comm: (?P<dump_comm>.*) (?:Not tainted|Tainted: (?P<dump_taint>[A-Z ]+)) (?P<dump_release>.*) (?P<dump_version>.*)',
'RIP: (?:[0-9a-f]+:)?(?:\[<(?P<dump_RIP>[0-9a-f]+)>\] )?(?: \[<[0-9a-f]+>\]  )?(?P<dump_function>\w+\+0x[0-9a-f]+/0x[0-9a-f]+(?: \[\w+\])?)',
'RSP: (?:[0-9a-f]+:)?(?P<dump_RSP>[0-9a-f]+) EFLAGS: (?P<dump_EFLAGS>[0-9a-f]+)',
'RAX: (?P<dump_RAX>[0-9a-f]+) RBX: (?P<dump_RBX>[0-9a-f]+) RCX: (?P<dump_RCX>[0-9a-f]+)',
'RDX: (?P<dump_RDX>[0-9a-f]+) RSI: (?P<dump_RSI>[0-9a-f]+) RDI: (?P<dump_RDI>[0-9a-f]+)',
'RBP: (?P<dump_RBP>[0-9a-f]+) R08: (?P<dump_R08>[0-9a-f]+) R09: (?P<dump_R09>[0-9a-f]+)',
'R10: (?P<dump_R10>[0-9a-f]+) R11: (?P<dump_R11>[0-9a-f]+) R12: (?P<dump_R12>[0-9a-f]+)',
'R13: (?P<dump_R13>[0-9a-f]+) R14: (?P<dump_R14>[0-9a-f]+) R15: (?P<dump_R15>[0-9a-f]+)',
'CR2: (?P<dump_CR2>[0-9a-f]+) CR3: (?P<dump_CR3>[0-9a-f]+) CR4: (?P<dump_CR4>[0-9a-f]+)',
' (?:\[<[0-9a-f]+>\] )?(?P<dump_stacktrace>\w+\+0x[0-9a-f]+/0x[0-9a-f]+(?: \[\w+\])?)',
'Code: (?P<dump_code>.*)',
'---\[ now (?P<dump_datetime>[0-9]+-[0-9]+-[0-9]+ [0-9]+:[0-9]+:[0-9]+\+[0-9]+)(?: at (?P<dump_host>.*))? \]---',

'(?P<info_pstore_dump>--- BEGIN OF PSTORE .*)',
'(?P<error_pstore_panic>Panic#(?P<dump_panic_nr>[0-9]+) Part1).*'
    ]

    __patterns_re = re.compile(r'^(?:(?P<log_prefix>.*) kernel: )?(?:<(?P<log_level>[0-9])>)?(?:\[ *(?P<log_uptime>[0-9]+\.[0-9]+)\] )?(?:' + '|'.join(__patterns) + ')$')

    __prefix_re = re.compile(r'<[0-9]>\[ *[0-9]+\.[0-9]+\] ')

    def __init__(self, textfile, max_lines=None, filename=None, hostname=None, date=None, print_raw=None):
        self.textfile = textfile
        self.max_lines = max_lines
        self.filename = filename
        self.hostname = hostname
        self.date = date
        self.line_nr = 0
        self.print_raw = print_raw

    def split_lines(self):
        while True:
            raw_line = self.textfile.readline()
            if not raw_line:
                break
            pos = 0
            self.line_nr += 1
            for match in self.__prefix_re.finditer(raw_line):
                start = match.start()
                if pos != start:
                    yield raw_line[pos:start]
                pos = start
            if pos != len(raw_line):
                yield raw_line[pos:]

    def events(self):
        release = '?'
        uptime = None
        event = {'text': ''}
        for raw_line in self.split_lines():
            if self.max_lines and self.line_nr > self.max_lines:
                yield {'severity': 'error',
                       'type': 'log_too_long',
                       'message': 'log too long, stop at line ' + str(self.line_nr),
                       'line': self.line_nr,
                       'host': self.hostname,
                       'file': self.filename,
                       'release': release,
                       'text': ''}
                break

            match = self.__patterns_re.match(raw_line)

            if match is None:
                if self.print_raw == 'all':
                    print(raw_line.rstrip())

                if 'severity' in event and self.line_nr > event['line'] + 100:
                    if 'release' not in event:
                        event['release'] = release
                    yield event
                    event = {'text': ''}
                continue

            new_event = False

            for key, val in match.groupdict().items():
                if val is None:
                    continue
                if key in ['dump_release', 'dump_boot_release', 'dump_release_hung_task']:
                    release = val
                severity, key = key.split('_', 1)
                if severity == 'log':
                    if key == 'uptime':
                        uptime = float(val)
                elif severity == 'dump':
                    if key == 'stacktrace':
                        if key not in event:
                            event[key] = []
                        event[key].append(val)
                        continue
                    if key not in event:
                        event[key] = val
                    elif val not in event[key].splitlines():
                        event[key] += '\n' + val
                elif severity == 'cont' and 'line' in event and self.line_nr < event['line'] + 100:
                    event['message'] += '\n' + val
                else:
                    if 'severity' in event:
                        if 'release' not in event:
                            event['release'] = release
                        yield event
                    event = {'severity': severity,
                             'type': key,
                             'message': val,
                             'line': self.line_nr,
                             'text': ''}
                    if self.filename:
                        event['file'] = self.filename
                    if self.hostname:
                        event['host'] = self.hostname
                    if self.date:
                        event['date'] = self.date
                    if uptime:
                        event['uptime'] = uptime
                    new_event = True

            event['text'] += raw_line

            if self.print_raw and (self.print_raw != 'warn' or event.get('severity', 'info') != 'info'):
                if new_event:
                    print('\n#', release, event['severity'], event['type'], 'at', self.hostname, self.date, '\n')
                print(raw_line.rstrip())

        if 'severity' in event:
            if 'release' not in event:
                event['release'] = release
            yield event


class EventReport(object):

    __versplit_re = re.compile(r'[^0-9]+')

    __filename_re = re.compile(r'^(?P<host>[^/]+)-(?P<date>[0-9]+-[0-9]{2}-[0-9]{2})\.log$')

    def __init__(self, max_lines=None, print_raw=None, progress=False, sources=[], coroner_url=CORONER_API):
        self.all_events = []
        self.events = {}
        self.max_lines = max_lines
        self.print_raw = print_raw
        self.progress = progress
        self.sources = sources
        self.today = datetime.date.today().isoformat()
        self.coroner_url = coroner_url

    def add_source(self, src, filename, hostname=None, date=None):
        if hostname is None and filename is not None:
            m = self.__filename_re.match(filename)
            if m and hostname is None:
                hostname = m.group('host')
            if m and date is None:
                date = m.group('date')
        self.sources.append((src, filename, hostname, date))

    def group(self, event):
        return (event['severity'], event['release'], event['type'])

    def group_rank(self, group):
        severity = ['info', 'warn', None, 'cont', 'error'].index(group[0])
        release = [int(x) for x in self.__versplit_re.split(group[1]) if x]
        return (severity, release, group[2])

    def ranked_groups(self):
        groups = list(self.events.keys())
        groups.sort(key=self.group_rank)
        return groups

    def parse_log(self, log, filename=None, hostname=None, date=None):
        parser = EventParser(log, filename=filename, hostname=hostname, date=date,
                             max_lines=self.max_lines, print_raw=self.print_raw)
        for event in parser.events():
            self.all_events.append(event)
            group = self.group(event)
            if group not in self.events:
                self.events[group] = []
            self.events[group].append(event)

    def read_events(self, log, filename=None, hostname=None, date=None):
        try:
            for event in json.load(log):
                self.all_events.append(event)
                group = self.group(event)
                if group not in self.events:
                    self.events[group] = []
                self.events[group].append(event)
        except Exception as e:
            print("fail to load events from: {}, err:{}".format(filename, str(e)), file=sys.stderr)

    def parse(self):
        reader = codecs.getreader('latin1')
        progress = 0
        total = len(self.sources)
        for src, filename, hostname, date in self.sources:
            op = self.parse_log
            try:
                if src == 'dmesg':
                    size = ''
                    proc = subprocess.Popen(['dmesg'], stdout=subprocess.PIPE)
                    log = reader(proc.stdout)
                elif src == 'file':
                    size = os.path.getsize(filename)
                    log = reader(open(filename, 'rb'))
                elif src == 'zcat':
                    size = os.path.getsize(filename)
                    p = subprocess.Popen(['zcat', filename], stdout=subprocess.PIPE)
                    log = reader(p.stdout)
                elif src == 'coroner':
                    res = requests.get(self.coroner_url + '/' + filename, stream=True)

                    if res.status_code != 200 and date[:-3] != self.today[:-3]:
                        res = requests.get(self.coroner_url + '/' + date[:-3] + '/' + filename, stream=True)

                    if res.status_code == 200:
                        log = reader(res.raw)
                        size = res.headers['Content-length']
                    else:
                        log = None
                        size = ''
                elif src == 'remote_dmesg':
                    size = ''
                    proc = subprocess.Popen(['ssh', '-n', hostname, 'dmesg'], stdout=subprocess.PIPE)
                    log = reader(proc.stdout)
                elif src == 'remote_file':
                    size = ''
                    proc = subprocess.Popen(['ssh', '-n', hostname, 'tail', '-n', str(self.max_lines), filename], stdout=subprocess.PIPE)
                    log = reader(proc.stdout)
                elif src == "json":
                    size = os.path.getsize(filename)
                    log = reader(open(filename, 'rb'))
                    op = self.read_events
                else:
                    print("unknown source", src)
            except Exception as e:
                log = None
                size = 'err: ' + str(e)

            if self.progress:
                progress += 100
                print("{:3}% {} {} {} {}".format(progress // total, hostname, date, filename, size),  file=sys.stderr)

            if log is not None:
                op(log, filename=filename, hostname=hostname, date=date, )

    def sample_strings(self, strings, count):
        if len(strings) <= count:
            return strings
        result = []
        for index in range(0, len(strings)-1, (len(strings)+count-1) // count):
            result.append(strings[index])
        return result

    def dump(self):
        print()
        for group in self.ranked_groups():
            hosts = []
            files = []
            messages = []
            severity, release, event_type = group
            for event in self.events[group]:
                if 'host' in event and event['host'] not in hosts:
                    hosts.append(event['host'])
                if 'file' in event and event['file'] not in files:
                    files.append(event['file'])
                message = event['message']
                if 'function' in event:
                    message += ' in ' + str(event['function'])
                if message not in messages:
                    messages.append(message)

            print('{:12} {:4} {:5} {:16} at {:3} hosts\n'.format(
                  release, len(self.events[group]), severity, event_type, len(hosts)))

            for message in self.sample_strings(messages, 10):
                print('  ' + message)
            print()

            for file in self.sample_strings(files, 100):
                print('  ' + file)
            print()

        print()


def main():
    coroner_logs = None
    if os.path.exists(CORONER_LOGS):
        coroner_logs = CORONER_LOGS

    parser = argparse.ArgumentParser(description='Coroner kernel log parser')

    parser.add_argument('-H', '--host', action='append', default=[], metavar='HOST_NAME', help='add host into request, default: localhost')
    parser.add_argument('-F', '--hosts-file', action='append', default=[], metavar='HOSTS_FILE', help='read host names from file')
    parser.add_argument('-w', '--walle-tag', action='append', default=[], metavar='WALLE_TAG', help='add all hosts from walle tag')
    parser.add_argument('-W', '--walle-project', action='append', default=[], metavar='WALLE_PROJECT', help='add all hosts from walle project')
    parser.add_argument('--domain', default=DEFAULT_DOMAIN, help='append domain to non-FQDN names, default: %(default)s')
    parser.add_argument('--list-hosts', action='store_true', help='list hosts and exit')
    parser.add_argument('-p', '--progress', action='store_true', help='print progress')
    parser.add_argument('-d', '--date', action='append', default=[], metavar='DATE', help='parse logs for this date %%Y-%%m-%%d, default: today')
    parser.add_argument('-D', '--days', type=int, default=2, metavar='DAYS', help='parse logs for days before each date, default: %(default)s')
    parser.add_argument('-C', '--coroner-logs', metavar='PATH', help='path to local coroner logs, default: %(default)s', default=coroner_logs)
    parser.add_argument('--coroner-url', metavar='URL', help='default: %(default)s', default=CORONER_API)
    parser.add_argument('-L', '--logs', action='store_true', help='parse logs from dmesg and kern.log')
    parser.add_argument('-m', '--max-lines', type=int, default=100000, help='limit for long logs, default: %(default)s')
    parser.add_argument('-r', '--raw', dest='print_raw', action='store_const', const='warn', help='dump raw warning and error events')
    parser.add_argument('-R', '--raw-info', dest='print_raw', action='store_const', const='info', help='dump all matched raw lines')
    parser.add_argument('-A', '--raw-all', dest='print_raw', action='store_const', const='all', help='dump all raw lines')
    parser.add_argument('-j', '--json', dest='print_json', action='store_const', const='events', help='dump all events as json')
    parser.add_argument('-e', '--events', action='append', default=[], metavar='EVENT_FILE', help='read events from file')
    parser.add_argument('files', nargs='*', metavar='FILE', help='parse log from files')

    args = parser.parse_args()

    report = EventReport(max_lines=args.max_lines, print_raw=args.print_raw, progress=args.progress, coroner_url=args.coroner_url)

    hosts = args.host

    for fn in args.hosts_file:
        hosts += [line.strip() for line in open(fn)]

    if args.domain:
        hosts = [host if '.' in host else host + '.' + args.domain for host in hosts]

    for project in args.walle_project:
        hosts += [h['name'] for h in walle_hosts(project=project, fields='name')]

    for tag in args.walle_tag:
        hosts += [h['name'] for h in walle_hosts(tags=tag, fields='name')]

    localhost = socket.gethostname()
    if not hosts and not args.files and not args.walle_project and not args.walle_tag and not args.events:
        hosts.append(localhost)
        args.logs = True

    hosts.sort()

    if args.list_hosts:
        for host in hosts:
            print(host)
        sys.exit(0)

    dates = []
    for d in args.date or [report.today]:
        date = datetime.datetime.strptime(d, '%Y-%m-%d').date()
        for n in range(-args.days, 1):
            dates.append((date + datetime.timedelta(days=n)).isoformat())

    if not hosts and not args.logs and not args.files and not args.events:
        parser.print_help()
        sys.exit(1)

    for filename in args.events:
        if not os.path.exists(filename):
            print('file not found', filename)
        report.add_source('json', filename)

    for filename in args.files:
        if not os.path.exists(filename):
            print('file not found', filename)
        elif filename.endswith('.gz'):
            report.add_source('zcat', filename)
        else:
            report.add_source('file', filename)

    prev_hostname = None
    for hostname in hosts:
        if hostname == prev_hostname:
            continue
        prev_hostname = hostname
        for date in dates:
            filename = '{}-{}.log'.format(hostname, date)
            if args.coroner_logs:
                coroner_log = os.path.join(args.coroner_logs, filename)
                if os.path.exists(coroner_log):
                    report.add_source('file', coroner_log, hostname, date)
            else:
                report.add_source('coroner', filename, hostname, date)
        if args.logs:
            if hostname == localhost:
                report.add_source('dmesg', 'dmesg', hostname, 'now')
                for fn in os.listdir('/var/log'):
                    if fn.startswith('kern.log') and os.path.isfile('/var/log/' + fn):
                        filename = '/var/log/' + fn
                        if fn.endswith('.gz'):
                            report.add_source('zcat', filename, hostname, '')
                        else:
                            report.add_source('file', filename, hostname, '')
            else:
                report.add_source('remote_dmesg', 'dmesg', hostname, 'now')
                report.add_source('remote_file', '/var/log/kern.log', hostname, '')

    report.parse()

    if args.print_json == 'events':
        print_json(report.all_events)
        print()
    else:
        report.dump()


if __name__ == "__main__":
    main()
