#!/usr/bin/env python3

import subprocess
import contextlib
import tempfile
import argparse
import shutil
import sys
import os


@contextlib.contextmanager
def tmpdir(**kwargs):
    tmp = tempfile.mkdtemp(**kwargs)
    try:
        yield tmp
    finally:
        shutil.rmtree(tmp)


# https://github.com/axboe/fio/issues/638
# https://github.com/axboe/fio/blob/master/.travis.yml
BUILD_DEPS = [
    'build-essential',
    'git',
    'libaio-dev',
    'libc6-dev',
    'libibverbs-dev',
    'libnuma-dev',
    'librbd-dev',
    'librdmacm-dev',
    'libz-dev',
    'zlib1g-dev'
]

def run(args, **kwargs):
    print("+ '" + "' '".join(args) + "'", flush=True)
    return subprocess.check_call(args, **kwargs)


def install_deps(args):
    cmd = ['sudo', 'apt-get', 'install', '--no-install-recommends', '-y'] + BUILD_DEPS
    if args.dry_run:
        print("# Install build dependencies.")
        print(" ".join(cmd))
        sys.exit(0)
    else:
        run(['sudo', 'apt-get', 'update', '-y'])
        run(cmd)


def build_bin(args):
    if args.install_deps:
        install_deps(args)

    with tmpdir(prefix='build-' + os.path.basename(args.src)) as build_dir:
        if os.path.isdir(args.src):
            src_dir = os.path.abspath(args.src)
        else:
            run(['tar', '--strip-components=1', '-xf', os.path.abspath(args.src)], cwd=build_dir)
            src_dir = build_dir

        help = subprocess.check_output([src_dir + '/configure', '--help'], cwd=build_dir)
        run([src_dir + '/configure',
             '--disable-native',    # remove -march=native to build portable binary
             '--build-static',
             '--prefix=/'], cwd=build_dir)
        run(['make', '-j', str(os.sysconf('SC_NPROCESSORS_ONLN'))], cwd=build_dir)

        if os.path.isdir(args.out):
            run(['make', 'install', 'DESTDIR=' + os.path.abspath(args.out) + '/fio'], cwd=build_dir)
        else:
            with tmpdir(prefix='out-' + os.path.basename(args.src)) as out_dir:
                run(['make', 'install', 'DESTDIR=' + os.path.abspath(out_dir) + '/fio'], cwd=build_dir)
                run(['tar', '-caf', os.path.abspath(args.out), 'fio'], cwd=out_dir)

if __name__ == '__main__':
    parser = argparse.ArgumentParser()
    subparsers = parser.add_subparsers(title="Possible extra actions", dest='command')

    deps = subparsers.add_parser(name="install-deps")
    deps.set_defaults(handle=install_deps)
    deps.add_argument("--dry-run", default=False, action='store_true')

    build = subparsers.add_parser(name="build")
    build.set_defaults(handle=build_bin)
    build.add_argument("--dry-run", default=False, action='store_true')
    build.add_argument("--install-deps", default=False, action='store_true')
    build.add_argument("src", help="directory or tarball")
    build.add_argument("out", help="directory or tarball")

    args = parser.parse_args()
    args.handle(args)
