#!/usr/bin/env python3

import subprocess
import contextlib
import tempfile
import argparse
import shutil
import sys
import os


@contextlib.contextmanager
def tmpdir(**kwargs):
    tmp = tempfile.mkdtemp(**kwargs)
    try:
        yield tmp
    finally:
        shutil.rmtree(tmp)


BUILD_DEPS = [
    'build-essential',
    'flex',
    'bison',
    'zlib1g-dev',
    'libbz2-dev',
    'liblzma-dev',
    'libzstd-dev',
    'libssl-dev',
    'binutils-dev',
    'libiberty-dev',
    # 'libdw-dev',
    # 'libelf-dev',
    'libnuma-dev',
    'libunwind-dev',
    'libnewt-dev',
    'libaudit-dev',
    'libslang2-dev',
    'asciidoc',
    'xmlto',
    'libpfm4',
    'libpfm4-dev'
]

def run(args, **kwargs):
    print("+ '" + "' '".join(args) + "'", flush=True)
    return subprocess.check_call(args, **kwargs)


def install_deps(args):
    cmd = ['sudo', 'apt-get', 'install', '--no-install-recommends', '-y'] + BUILD_DEPS
    if not args.elfutils:
        cmd += ['libdw-dev', 'libelf-dev']
    if args.dry_run:
        print("# Install build dependencies.")
        print(" ".join(cmd))
        sys.exit(0)
    else:
        run(['sudo', 'apt-get', 'update', '-y'])
        run(cmd)


def build_bin(args):
    if args.install_deps:
        install_deps(args)

    with tmpdir(prefix='build-' + os.path.basename(args.src)) as build_dir:
        run(['tar', '-xf', os.path.abspath(args.src), '--strip-components=1'], cwd=build_dir)
        tool_dir = build_dir + '/tools/perf'

        make = ['make', '-j', str(os.sysconf('SC_NPROCESSORS_ONLN')),
                'LDFLAGS=-static', 'VF=1', 'NO_LIBPERL=1', 'NO_LIBPYTHON=1', 'LIBPFM4=1']

        if args.elfutils:
            eu_build_dir = build_dir + '/elfutils'
            eu_dest_dir = build_dir + '/elfutils-out'
            os.mkdir(eu_build_dir)
            os.mkdir(eu_dest_dir)
            run(['tar', '--strip-components=1', '-xf', os.path.abspath(args.elfutils)], cwd=eu_build_dir)
            # run(['autoreconf', '-fis'], cwd=eu_build_dir)
            run(['./configure', '--disable-debuginfod', '--prefix=/usr'], cwd=eu_build_dir)
            run(['make', '-j', str(os.sysconf('SC_NPROCESSORS_ONLN'))], cwd=eu_build_dir)
            run(['make', 'install', 'DESTDIR=' + eu_dest_dir], cwd=eu_build_dir)
            # make += ['LIBDW_DIR=' + eu_dest_dir + '/usr']
            make += ['LDFLAGS=-static -L' + eu_dest_dir + '/usr/lib',
                     'EXTRA_CFLAGS=-I' + eu_dest_dir + '/usr/include']
            # elfutils 0.178 libdw.a contains backends but needs pthread
            run(['sed', 's/ -lebl / -lpthread /', '-i', build_dir + '/tools/perf/Makefile.config'])
            run(['sed', 's/ -lebl / -lpthread /', '-i', build_dir + '/tools/build/feature/Makefile'])

        run(make + ['clean'], cwd=tool_dir)
        run(make, cwd=tool_dir)
        with tmpdir(prefix='out-' + os.path.basename(args.src)) as out_dir:
            run(make + ['install', 'DESTDIR=' + os.path.abspath(out_dir)], cwd=tool_dir)
            run(['ln', 'libexec/perf-core/perf-archive', '.'], cwd=out_dir)
            run(['ln', 'bin/perf', '.'], cwd=out_dir)
            run(['./perf', '-vv'], cwd=out_dir)
            run(['tar', '-caf', os.path.abspath(args.out), '.'], cwd=out_dir)

if __name__ == '__main__':
    parser = argparse.ArgumentParser()
    subparsers = parser.add_subparsers(title="Possible extra actions", dest='command')
    deps = subparsers.add_parser(name="install-deps")
    deps.set_defaults(handle=install_deps)
    deps.add_argument("--dry-run", default=False, action='store_true')
    build = subparsers.add_parser(name="build")
    build.set_defaults(handle=build_bin)
    build.add_argument("--dry-run", default=False, action='store_true')
    build.add_argument("--install-deps", default=False, action='store_true')
    build.add_argument("--elfutils", help="elfutils source tarball")
    build.add_argument("src", help="source tarball")
    build.add_argument("out", help="output tarball")
    args = parser.parse_args()
    args.handle(args)
