#!/usr/bin/env python3

import subprocess
import contextlib
import tempfile
import argparse
import shutil
import sys
import os


@contextlib.contextmanager
def tmpdir(**kwargs):
    tmp = tempfile.mkdtemp(**kwargs)
    try:
        yield tmp
    finally:
        shutil.rmtree(tmp)


BUILD_DEPS = [
    'build-essential',
    'gawk',
    'autoconf',
    'automake',
    'git',
]

def run(args, **kwargs):
    print("+ '" + "' '".join(args) + "'", flush=True)
    return subprocess.check_call(args, **kwargs)


def install_deps(args):
    cmd = ['sudo', 'apt-get', 'install', '--no-install-recommends', '-y'] + BUILD_DEPS
    if args.dry_run:
        print("# Install build dependencies.")
        print(" ".join(cmd))
        sys.exit(0)
    else:
        run(['sudo', 'apt-get', 'update', '-y'])
        run(cmd)


def build_bin(args):
    if args.install_deps:
        install_deps(args)

    with tmpdir(prefix='build-' + os.path.basename(args.src)) as build_dir:
        run(['tar', '--strip-components=1', '-xf', os.path.abspath(args.src)], cwd=build_dir)
        run([build_dir + '/bootstrap'], cwd=build_dir)
        run([build_dir + '/configure', '--disable-mpers'], cwd=build_dir)
        run(['sed', '-i', 's/^CFLAGS = -g -O2/CFLAGS = -g -O2 -static/g', 'Makefile'], cwd=build_dir)
        run(['sed', '-i', 's/^LDFLAGS =/LDFLAGS = -pthread/g', 'Makefile'], cwd=build_dir)
        run(['make', '-j', str(os.sysconf('SC_NPROCESSORS_ONLN'))], cwd=build_dir)
        run(['tar', '-caf', os.path.abspath(args.out), 'strace', '--transform', 's,^,strace/,'], cwd=build_dir)

if __name__ == '__main__':
    parser = argparse.ArgumentParser()
    subparsers = parser.add_subparsers(title="Possible extra actions", dest='command')
    deps = subparsers.add_parser(name="install-deps")
    deps.set_defaults(handle=install_deps)
    deps.add_argument("--dry-run", default=False, action='store_true')
    build = subparsers.add_parser(name="build")
    build.set_defaults(handle=build_bin)
    build.add_argument("--dry-run", default=False, action='store_true')
    build.add_argument("--install-deps", default=False, action='store_true')
    build.add_argument("src", help="source tarball")
    build.add_argument("out", help="output tarball")
    args = parser.parse_args()
    args.handle(args)
