#!/bin/bash

DEPENDENCIES="api kernel library"

set -e

require_python()
{
    if [ -z "${PYTHON}" ]; then
        echo "Please export \$PYTHON variable!"
        exit 1
    else
        true
    fi
}

do_info()
{
    echo " * " $*
}

do_cleanup()
{
    do_info "cleanup $*"

    rm -rf ./dist ./*.egg-info

    if [ "$1" = "full" ]; then
        rm -rf ./build ./snapshot ./tmp
    fi

    find . -name '*.py[co]' -delete
}

do_prepare()
{
    do_info "preparing directories"
    mkdir -p ./snapshot/lib ./snapshot/bin
}

do_ctl()
{
    do_info "adding keychain.scsd $1"

    if [ "$1" = "symlink" ]; then
        ln -sf ../keychain.scsd snapshot
    else
        cp -f keychain.scsd snapshot
    fi

    $PYTHON -m compileall -q snapshot
}

do_etc()
{
    do_info "adding etc"
    rm -rf snapshot/etc
    mkdir snapshot/etc

    # if [ "x$1" = xsymlink ]; then
    #     ln -sf ../etc snapshot/etc
    # else
    #     cp -fr etc snapshot/etc
    # fi

    # $PYTHON -m compileall -q snapshot
}

do_call_setuppy()
{
    do_info "running: setup.py $*"
    PYTHONPATH=./snapshot/lib:$PYTHONPATH $PYTHON setup.py -q "$@"
}

do_bundle()
{
    require_python
    do_cleanup full
    do_prepare
    do_ctl
    do_etc
    do_call_setuppy egg_info bdist_egg --exclude-source-files
    do_call_setuppy easy_install -d ./snapshot/lib -s ./snapshot/bin -z -O2 -m dist/*.egg

    for f in ./snapshot/bin/*; do
        tail -n+2 "${f}" > "${f}.tmp"
        echo '#!/skynet/python/bin/python' > "${f}"
        cat "${f}.tmp" >> "${f}"
        rm -rf "${f}.tmp"
        chmod +x "${f}"
    done
    do_cleanup
}

do_develop()
{
    require_python
    do_cleanup
    do_prepare
    do_ctl symlink
    do_etc symlink
    do_call_setuppy develop -d ./snapshot/lib -s ./snapshot/bin -z -O2 -m

    rm -rf snapshot/ve

    local _egg_info=$(ls -d *.egg-info)
    mv "${_egg_info}" snapshot/ve

    local _egg_link=$(basename $(ls snapshot/lib/*.egg-link))
    echo "$(pwd)/snapshot/ve" > snapshot/lib/${_egg_link}
    echo "../.." >> snapshot/lib/${_egg_link}

    ln -s ../../src ./snapshot/ve/skynet_keychain
    do_cleanup
}

do_test()
{
    require_python
    mkdir -p tmp
    do_call_setuppy test "$@"
    do_cleanup
}

do_pylint()
{
    require_python
    PYTHONPATH=./snapshot/lib:$PYTHONPATH $PYTHON $(dirname $PYTHON)/pylint --rcfile=./pylint.conf skynet_keychain "$@"
}

do_run()
{
    PYTHONPATH=$(pwd)/snapshot/lib:$PYTHONPATH $PYTHON ./snapshot/bin/keychain "$@"
}

do_tgz()
{
    do_bundle
    rm -rf keychain.tgz
    tar -C snapshot -czvf keychain.tgz .
}

case "$1" in
    bundl*)
        do_bundle
        ;;

    dev*)
        do_develop
        ;;

    test*)
        shift
        do_test "$@"
        ;;

    pylint*)
        shift
        do_pylint "$@"
        ;;

    run*)
        shift 1
        do_run "$@"
        ;;

    tgz*)
        shift
        do_tgz "$@"
        ;;

    clean*)
        do_cleanup full
        ;;

    dep*)
        echo ${DEPENDENCIES}
        ;;

    *)
        echo "Usage $0 [options] develop | bundle | test | clean | deps | help"
        ;;
esac

