package server

import (
	"a.yandex-team.ru/infra/kube-auth-webhook/pkg/user"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/yandex/blackbox"
	"context"
	"encoding/json"
	"fmt"
	"io/ioutil"
	v1 "k8s.io/api/authentication/v1"
	"net/http"
	"time"
)

type Server struct {
	l      log.Logger
	yaAuth *user.YandexAuth
}

func New(l log.Logger, bb blackbox.Client, ipaddr string) (*Server, error) {
	yaAuth, err := user.NewYandexAuth(ipaddr, bb)
	if err != nil {
		return nil, err
	}
	return &Server{l, yaAuth}, nil
}

func (s *Server) AuthHandler(w http.ResponseWriter, r *http.Request) {
	ctx, cancel := context.WithTimeout(r.Context(), 10*time.Second)
	defer cancel()
	// Read body of POST request
	b, err := ioutil.ReadAll(r.Body)
	if err != nil {
		writeError(s.l, w, err, v1.TokenReview{})
		return
	}
	s.l.Infof("Received auth request from %s", r.RemoteAddr)
	// Unmarshal JSON from POST request to TokenReview object
	tr := v1.TokenReview{}
	err = json.Unmarshal(b, &tr)
	if err != nil {
		writeError(s.l, w, err, tr)
		return
	}
	token := tr.Spec.Token
	userInfo, err := s.yaAuth.UserInfo(ctx, token)
	if err != nil {
		writeError(s.l, w, err, tr)
		return
	}
	tr.Status.Authenticated = true
	tr.Status.User = *userInfo
	// Marshal the TokenReview to JSON and send it back
	b, err = json.Marshal(tr)
	if err != nil {
		writeError(s.l, w, err, tr)
		return
	}
	_, _ = w.Write(b)
	s.l.Infof("Successfully authenticated client from %s", r.RemoteAddr)
}

func writeError(l log.Logger, w http.ResponseWriter, err error, tr v1.TokenReview) {
	tr.Status.Error = err.Error()
	l.Error(err.Error())
	writeTr(w, tr)
}

func writeTr(w http.ResponseWriter, tr v1.TokenReview) {
	b, err := json.Marshal(tr)
	if err != nil {
		w.WriteHeader(http.StatusInternalServerError)
		_, _ = fmt.Fprintln(w, err)
	} else {
		_, _ = w.Write(b)
	}
}
