package user

import (
	"a.yandex-team.ru/infra/kube-auth-webhook/pkg/oauth"
	"a.yandex-team.ru/library/go/yandex/blackbox"
	"bytes"
	"context"
	"encoding/json"
	"fmt"
	"io"
	v1 "k8s.io/api/authentication/v1"
	"net/http"
	"os"
	"strings"
)

func NewYandexAuth(ipaddr string, bb blackbox.Client) (*YandexAuth, error) {
	clientID, ok := os.LookupEnv("OAUTH_CLIENT_ID")
	if !ok {
		clientID = oauth.ClientID
	}
	scopesStr, ok := os.LookupEnv("OAUTH_CLIENT_SCOPES")
	scopes := make([]string, 0)
	if ok {
		for _, scope := range strings.Split(scopesStr, ",") {
			scopes = append(scopes, strings.TrimSpace(scope))
		}
	}
	staffToken := os.Getenv("STAFF_TOKEN")
	if staffToken == "" {
		return nil, fmt.Errorf("failed to find environment variable STAFF_TOKEN")
	}
	return &YandexAuth{
		ipaddr,
		bb,
		clientID,
		scopes,
		staffToken,
	}, nil
}

type YandexAuth struct {
	ipaddr         string
	bb             blackbox.Client
	clientID       string
	scopes         []string
	rootStaffToken string
}

func (a *YandexAuth) UserInfo(ctx context.Context, token string) (*v1.UserInfo, error) {
	user, err := a.User(ctx, token)
	if err != nil {
		return nil, err
	}
	groups, err := a.Groups(user.Username)
	if err != nil {
		return nil, err
	}
	groupsList := make([]string, 0, len(groups))
	for _, g := range groups {
		groupsList = append(groupsList, g.Group.URL)
	}
	return &v1.UserInfo{
		Username: user.Username,
		UID:      fmt.Sprintf("%d", user.UID),
		Groups:   groupsList,
		Extra:    map[string]v1.ExtraValue{},
	}, nil
}

func (a *YandexAuth) Groups(username string) ([]*Group, error) {
	er := func(err error) error {
		return fmt.Errorf("failed to get groups info from staff: %w", err)
	}
	respStruct := &struct {
		Links  struct{} `json:"links"`
		Page   int      `json:"page"`
		Limit  int      `json:"limit"`
		Result []*Group `json:"result"`
		Total  int      `json:"total"`
		Pages  int      `json:"pages"`
	}{}
	url := fmt.Sprintf("https://staff-api.yandex-team.ru/v3/groupmembership?person.login=%s&group.type=department,service,servicerole&_limit=200", username)
	req, err := http.NewRequest("GET", url, &bytes.Buffer{})
	if err != nil {
		return nil, er(err)
	}
	// May be we need use tvm ticket here
	req.Header.Set("Authorization", fmt.Sprintf("OAuth %s", a.rootStaffToken))
	resp, err := http.DefaultClient.Do(req)
	if err != nil {
		return nil, er(err)
	}
	respBytes, err := io.ReadAll(resp.Body)
	if err != nil {
		return nil, er(err)
	}
	if resp.StatusCode >= 400 {
		return nil, er(fmt.Errorf("%d status code, body: '%s'", resp.StatusCode, respBytes))
	}
	if err := json.Unmarshal(respBytes, respStruct); err != nil {
		return nil, er(fmt.Errorf("'%w', response: '%s'", err, respBytes))
	}
	return respStruct.Result, nil
}

func (a *YandexAuth) checkScopes(scopes []string) bool {
	scopesMap := make(map[string]bool)
	for _, scope := range scopes {
		scopesMap[scope] = true
	}
	for _, scope := range a.scopes {
		if !scopesMap[scope] {
			return false
		}
	}
	return true
}

func (a *YandexAuth) User(ctx context.Context, token string) (*User, error) {
	bbResponse, err := a.bb.OAuth(ctx, blackbox.OAuthRequest{
		OAuthToken: token,
		UserIP:     a.ipaddr,
	})
	if err != nil {
		return nil, fmt.Errorf("failed to get user from blackbox: %w", err)
	}
	if len(a.scopes) != 0 && a.checkScopes(bbResponse.Scopes) {
		return &User{Username: bbResponse.User.Login, UID: int(bbResponse.User.UID.ID)}, nil
	}
	if bbResponse.ClientID != a.clientID {
		return nil, fmt.Errorf("invalid token issuer (%s): clientID='%s'", bbResponse.ClientName, bbResponse.ClientID)
	}
	return &User{Username: bbResponse.User.Login, UID: int(bbResponse.User.UID.ID)}, nil
}

// Group from staff response format
// Commented not used fields
type Group struct {
	//JoinedAt time.Time `json:"joined_at"`
	//ID       int       `json:"id"`
	Group struct {
		//IsDeleted bool   `json:"is_deleted"`
		//Name      string `json:"name"`
		//Service   struct {
		//	ID int `json:"id"`
		//} `json:"service"`
		URL string `json:"url"`
		//Department struct {
		//	ID int `json:"id"`
		//} `json:"department"`
		//Type string `json:"type"`
		//ID   int    `json:"id"`
	} `json:"group"`
	//Person struct {
	//	IsDeleted bool   `json:"is_deleted"`
	//	UID       string `json:"uid"`
	//	Official  struct {
	//		Affiliation  string `json:"affiliation"`
	//		IsDismissed  bool   `json:"is_dismissed"`
	//		IsRobot      bool   `json:"is_robot"`
	//		IsHomeworker bool   `json:"is_homeworker"`
	//	} `json:"official"`
	//	Login string `json:"login"`
	//	ID    int    `json:"id"`
	//	Name  struct {
	//		Last struct {
	//			Ru string `json:"ru"`
	//			En string `json:"en"`
	//		} `json:"last"`
	//		First struct {
	//			Ru string `json:"ru"`
	//			En string `json:"en"`
	//		} `json:"first"`
	//	} `json:"name"`
	//} `json:"person"`
}

type User struct {
	Username string
	UID      int
}
