package controllers

import (
	corev1 "k8s.io/api/core/v1"
	"sigs.k8s.io/controller-runtime/pkg/event"
	"sigs.k8s.io/controller-runtime/pkg/predicate"
)

// isManaged checks if service object is managed by this controller.
// Currently, we use .Type which can be changed during object lifecycle, and
// we may leave garbage in YP if we miss such change event (which can happen).
// Seemingly better approach: encode users intern to expose in .Name which cannot
// be changed online, in case of removal/creation we can add ourselves in finalizer list.
func isManaged(s *corev1.Service) bool {
	return s.Spec.Type == corev1.ServiceTypeNodePort
}

var (
	// servicePredicate defines filters which check if our reconciler should be called -
	// e.g. if we have or had the annotation (actual check done in isManaged function above).
	// As reconciler does not have access to events (only reconcile.Request) - it cannot
	// check if previous version of the object has been exposed (i.e. managed by us).
	// I (nekto0n) found the only way to access events through predicates.
	// But: we in reconciler we cannot use this as an invariant, that only managed services
	// will trigger reconcile event as .Reconcile can be triggered because of Endpoints object change.
	servicePredicate = predicate.Funcs{
		CreateFunc: func(e event.CreateEvent) bool {
			obj := e.Object.(*corev1.Service)
			return isManaged(obj)
		},
		DeleteFunc: func(e event.DeleteEvent) bool {
			obj := e.Object.(*corev1.Service)
			return isManaged(obj)
		},
		UpdateFunc: func(e event.UpdateEvent) bool {
			// User could have removed annotation - we need to handle that too,
			// e.g. remove objects from YP.
			obj := e.ObjectNew.(*corev1.Service)
			if isManaged(obj) {
				return true
			}
			obj = e.ObjectOld.(*corev1.Service)
			return isManaged(obj)
		},
		GenericFunc: func(e event.GenericEvent) bool {
			obj := e.Object.(*corev1.Service)
			return isManaged(obj)
		},
	}
)
