/*
Copyright 2022.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package controllers

import (
	"a.yandex-team.ru/infra/kube/awacs-bridge/internal/ypsync"
	"context"
	"fmt"
	"k8s.io/apimachinery/pkg/types"
	"k8s.io/client-go/kubernetes"
	"k8s.io/client-go/tools/record"
	"sigs.k8s.io/controller-runtime/pkg/reconcile"

	corev1 "k8s.io/api/core/v1"
	apierrors "k8s.io/apimachinery/pkg/api/errors"
	ctrl "sigs.k8s.io/controller-runtime"
	"sigs.k8s.io/controller-runtime/pkg/builder"
	"sigs.k8s.io/controller-runtime/pkg/client"
	"sigs.k8s.io/controller-runtime/pkg/handler"
	"sigs.k8s.io/controller-runtime/pkg/log"
	"sigs.k8s.io/controller-runtime/pkg/source"
)

// ServiceReconciler reconciles a Service object
type ServiceReconciler struct {
	client.Client                      // Service client
	clientset     kubernetes.Interface // clientset to get endpoints and nodes
	recorder      record.EventRecorder // recorder to emit diagnostic events
	ypSync        ypsync.Interface     // yp synchronizer interface doing all heavy lifting syncing with YP
}

func NewReconciler(clientset kubernetes.Interface, c client.Client, r record.EventRecorder, y ypsync.Interface) *ServiceReconciler {
	return &ServiceReconciler{
		clientset: clientset,
		Client:    c,
		recorder:  r,
		ypSync:    y,
	}
}

func (r *ServiceReconciler) handleUpdate(ctx context.Context, s *corev1.Service) error {
	fmt.Printf("Managing exposed service '%s'...\n", s.Name)
	endpoints, err := ypsync.ExtractEndpoints(ctx, s, r.clientset)
	if err != nil {
		return err
	}
	return r.ypSync.Expose(ctx, endpoints)
}

func (r *ServiceReconciler) handleDelete(ctx context.Context, s *corev1.Service) error {
	return r.ypSync.Delete(ctx, types.NamespacedName{
		Namespace: s.Namespace,
		Name:      s.Name,
	})
}

// Reconcile is the main entry point into controller.
// Called by the underlying machinery when Service **or** Endpoins objects change.
func (r *ServiceReconciler) Reconcile(ctx context.Context, req ctrl.Request) (ctrl.Result, error) {
	l := log.FromContext(ctx).WithName(req.NamespacedName.String())
	// Get service object corresponding to the request
	obj := &corev1.Service{}
	if err := r.Client.Get(ctx, req.NamespacedName, obj); err != nil {
		if apierrors.IsNotFound(err) {
			// we'll ignore not-found errors, since we can get them on deleted requests.
			return ctrl.Result{}, nil
		}
		l.Error(err, "unable to fetch service")
		return ctrl.Result{}, err
	}
	// Service is being deleted - handle deletion.
	if !obj.DeletionTimestamp.IsZero() {
		l.Info("handling deletion...")
		return ctrl.Result{}, r.handleDelete(ctx, obj)
	}
	// Service is not considered managed - ensure not present in YP.
	if !isManaged(obj) {
		l.Info("handling update of not managed service")
		return ctrl.Result{}, r.handleDelete(ctx, obj)
	}
	l.Info("handling modification...")
	if err := r.handleUpdate(ctx, obj); err != nil {
		r.recorder.Eventf(obj, "Warning", "Invalid", "%s", err)
		return ctrl.Result{}, err
	}
	return ctrl.Result{}, nil
}

// endpointWatchEventFunc triggers reconcile request for corresponding service if endpoints change
func endpointWatchEventFunc(o client.Object) []reconcile.Request {
	// Side effect: this triggers reconcile for every endpoint set change, even
	// for non managed ones, which may be unfortunate.
	return []reconcile.Request{
		// Construct reconcile request with namespaced name, which we'll be used as name of service object
		// in .Reconcile
		{
			NamespacedName: types.NamespacedName{
				Name:      o.GetName(),
				Namespace: o.GetNamespace(),
			},
		},
	}
}

// SetupWithManager sets up the controller with the Manager.
func (r *ServiceReconciler) SetupWithManager(mgr ctrl.Manager) error {
	return ctrl.NewControllerManagedBy(mgr).
		// We watch services and gather a list services exposed by this controller.
		For(&corev1.Service{}, builder.WithPredicates(servicePredicate)).
		// We want to trigger reconcile request after Endpoints changes (e.g. service scaled),
		// thus we use map function to translate Endpoints event into reconcile request for corresponding Service
		Watches(&source.Kind{Type: &corev1.Endpoints{}},
			handler.EnqueueRequestsFromMapFunc(endpointWatchEventFunc)).
		Complete(r)
}
