package ypsync

import (
	"a.yandex-team.ru/yp/go/proto/ypapi"
	"context"
	"fmt"
	corev1 "k8s.io/api/core/v1"
	metav1 "k8s.io/apimachinery/pkg/apis/meta/v1"
	"k8s.io/apimachinery/pkg/types"
	"k8s.io/client-go/kubernetes"
	"strings"
)

const (
	// Yandex network prefix, according to experience and https://bgpview.io/prefix/2a02:6b8::/29
	yandexNetPrefix = "2a02:6b8"
	// Well-known label according to https://kubernetes.io/docs/reference/labels-annotations-taints/
	topologyLabel = "topology.kubernetes.io/zone"
)

// Endpoint is an internal representation, having enough information to compare with YP one.
type Endpoint struct {
	NodeName  string // Kubernetes node name (not fqdn), used as endpoint ID
	Port      int    // Service port (node port)
	Address   string // NodeName's IPv6 address
	ServiceID string // Service identifier. Used as endpoint set ID
	Protocol  string // Endpoints protocol: "TCP" or "UDP"
}

// ID generates YP endpoint identifier (as unique as I could think of).
func (e *Endpoint) ID() string {
	return fmt.Sprintf("%s:%s", e.ServiceID, e.NodeName)
}

// EqualTo compares endpoint with YP counterpart
func (e *Endpoint) EqualTo(p *ypapi.TEndpoint) bool {
	if p.Meta.ParentKey != e.ServiceID {
		return false
	}
	if p.Meta.Id != e.NodeName {
		return false
	}
	if *p.Spec.Ip6Address != e.Address {
		return false
	}
	if int(*p.Spec.Port) != e.Port {
		return false
	}
	return true
}

type Set struct {
	Name         types.NamespacedName
	TopologyZone string // Topology zone, e.g "ru-central1-a"
	Endpoints    []*Endpoint
}

func allSets(name types.NamespacedName) map[string]*Set {
	sets := make(map[string]*Set, len(zoneMap))
	for k := range zoneMap {
		sets[k] = &Set{Name: name, TopologyZone: k}
	}
	return sets
}

func ExtractEndpoints(ctx context.Context, s *corev1.Service, clientset kubernetes.Interface) (map[string]*Set, error) {
	endpoints, err := clientset.CoreV1().Endpoints(s.Namespace).Get(ctx, s.Name, metav1.GetOptions{})
	if err != nil {
		return nil, err
	}
	getNodeFun := clientset.CoreV1().Nodes().Get
	// Extract NodePort from service ports spec
	num := -1
	protocol := ""
	for _, port := range s.Spec.Ports {
		if port.NodePort != 0 {
			num = int(port.NodePort)
			protocol = string(port.Protocol)
			break
		}
	}
	sets := allSets(types.NamespacedName{Name: s.Name, Namespace: s.Namespace})
	if num == -1 {
		return nil, fmt.Errorf("failed to find non-zero NodePort in service")
	}
	// Extract endpoints going over all subsets
	for _, sub := range endpoints.Subsets {
		// According to the docs: we need cartesian product of addresses x ports,
		// but ports in Endpoints object are ports on which pods listen, not nodePort,
		// so we use port from Service.Spec
		for _, e := range sub.Addresses {
			// Get node object for this endpoint - as we need nodes' address.
			n, err := getNodeFun(ctx, *e.NodeName, metav1.GetOptions{})
			if err != nil {
				return nil, err
			}
			// Find Yandex IPv6 address in node addresses.
			// Seems like we do not have other discriminants in real life:
			//   * all addresses are marked as "Internal"
			//   * nodes can have multiple v6 addresses (link-local ones)
			addr := ""
			for _, a := range n.Status.Addresses {
				if strings.HasPrefix(a.Address, yandexNetPrefix) {
					addr = a.Address
					break
				}
			}
			// Extract node's zone to add it to appropriate set
			zone := n.Labels[topologyLabel]
			t, found := sets[zone]
			if !found {
				return nil, fmt.Errorf("node '%s' has unsupported label %s=%s", n.Name, topologyLabel, zone)
			}
			if addr == "" {
				return nil, fmt.Errorf("failed to find yandex address for node '%s'", n.Name)
			}
			// TODO: can we have duplicate endpoints? If so - use hashmap for endpoints.
			t.Endpoints = append(t.Endpoints, &Endpoint{
				NodeName:  n.Name,
				Port:      num,
				Address:   addr,
				ServiceID: endpoints.Name,
				Protocol:  protocol,
			})
		}
	}
	return sets, nil
}
