/*
	YP related utilities.
	Notes:
		* Uses YSON (kind of unstructured wire-level API) because I failed to create nested labels in proto notation.
*/

// TODO: logs, not fmt

package ypsync

import (
	"context"
	"fmt"
	"strings"

	"a.yandex-team.ru/yp/go/yp"
	"a.yandex-team.ru/yp/go/yp/yperrors"
	"a.yandex-team.ru/yp/go/yson/ypapi"
	"a.yandex-team.ru/yp/go/yson/ytypes"
	"a.yandex-team.ru/yt/go/yson"
	"a.yandex-team.ru/yt/go/yterrors"
	"a.yandex-team.ru/yt/yt/orm/go/orm/ormerrors"
	"google.golang.org/protobuf/proto"
	"k8s.io/apimachinery/pkg/types"
)

const (
	// Handy label to leave some breadcrumbs when someone finds these endpoints.
	kubeLabel  = "kube-bridge.service-id"
	labelValue = "awacs-bridge-sync"
)

var (
	readyStatus    = &ypapi.TEndpointStatus{Ready: proto.Bool(true)}
	endpointLabels = ytypes.AttributeDictionary{
		kubeLabel: labelValue,
	}
)

// nannyLabels generates labels which will fool awacs common balancer validation,
// thus we can use it for tests. see:
// https://bb.yandex-team.ru/projects/NANNY/repos/nanny/browse/nanny/src/nanny/model/yp_lite/ui_proxy/yp_endpoint_sets_ctl.py#161
func nannyLabels(name string) ytypes.AttributeDictionary {
	return ytypes.AttributeDictionary{
		"nanny": map[string]string{
			"nanny_service_id": name,
		},
	}
}

type YPSync struct {
	name   string
	client *ClientFactory
}

// New initializes YPSync object using provided name and YP client.
// Parameters:
//   * `name` - is supposedly short, but unique string
//     to construct YP endpoint IDs uniquely between multiple instances of this
//     controller. E.g. we could use something like 'taxi-prod'.
func New(name string) *YPSync {
	return &YPSync{name, NewClientFactory()}
}

// genSetID generates identifier for YP endpoint set
func (y *YPSync) genSetID(name types.NamespacedName) string {
	var sb strings.Builder
	defer sb.Reset()
	sb.WriteString(name.Name)
	sb.WriteString(".")
	sb.WriteString(name.Namespace)
	sb.WriteString(".")
	sb.WriteString(y.name)
	return sb.String()
}

func (y *YPSync) Delete(ctx context.Context, name types.NamespacedName) error {
	// As we have only name, we do not know in which cluster to delete - delete from all.
	for k := range zoneMap {
		client, err := y.client.For(k)
		if err != nil {
			return err
		}
		fmt.Printf("Removing %s from YP(%s)...\n", name.String(), k)
		id := y.genSetID(name)
		// Removing set will conveniently CASCADE remove endpoints.
		_, err = client.RemoveEndpointSet(ctx, yp.RemoveEndpointSetRequest{
			ID:                id,
			IgnoreNonexistent: true, // Do not error if no object exists
		})
		fmt.Println("Done!")
		if err != nil {
			return err
		}
	}
	return nil
}

func (y *YPSync) createEndpointSet(ctx context.Context, c *yp.Client, id string) error {
	_, err := c.GetObject(ctx, yp.GetObjectRequest{
		ObjectID:   id,
		ObjectType: yp.ObjectTypeEndpointSet,
		Format:     yp.PayloadFormatYson,
	})
	if err == nil { // Object already exists
		return nil
	}
	if !yterrors.ContainsErrorCode(err, yperrors.CodeNoSuchObject) {
		// Other kind of error, not "not found" one.
		return err
	}
	// Ensured that object doesn't exist - create
	_, err = c.CreateEndpointSet(ctx, yp.CreateEndpointSetRequest{
		EndpointSet: ypapi.TEndpointSet{
			Meta: &ypapi.TEndpointSetMeta{
				Id: &id,
			},
			Labels: nannyLabels(id),
		},
	})
	return err
}

// createEndpoints ensures that provided endpoints are present in corresponding YP
func (y *YPSync) createEndpoints(ctx context.Context, c *yp.Client, set *Set) error {
	id := y.genSetID(set.Name)
	// Select current endpoints using filter
	// TODO: Move to variable, and use it in strings.Builder
	filter := fmt.Sprintf(`[/meta/endpoint_set_id] = "%s"`, id)
	fmt.Println("Selecting endpoints matching ", filter)
	resp, err := c.SelectEndpoints(ctx, yp.SelectEndpointsRequest{
		Filter:            filter,
		Format:            yp.PayloadFormatYson,
		ContinuationToken: "", // TODO: add continuation token handling
		// TODO: wtf is continuation token?
	})
	if err != nil {
		return err
	}
	// TODO: Looks like for future use
	var currentSet []*ypapi.TEndpoint
	for resp.Next() {
		e := &ypapi.TEndpoint{}
		if err := resp.Fill(e); err != nil {
			return err
		}
		currentSet = append(currentSet, e)
	}
	_ = currentSet // get rid of unused for now

	fmt.Printf("Found %d endpoints in YP\n", resp.Count())
	req := yp.CreateEndpointRequest{}
	for _, e := range set.Endpoints {
		obj := &ypapi.TEndpoint{
			Meta: &ypapi.TEndpointMeta{
				Id:            proto.String(e.ID()),
				EndpointSetId: &id,
			},
			Spec: &ypapi.TEndpointSpec{
				Protocol:   &e.Protocol,
				Fqdn:       &e.NodeName, // Not a fqdn per-se, but should work - it is not used usually.
				Ip6Address: &e.Address,
				Port:       proto.Int32(int32(e.Port)),
			},
			// We need to set this status, otherwise awacs balancer will not use this endpoint.
			// In k8s if endpoint is present.
			Status: readyStatus,
			Labels: endpointLabels, // Add some labels to be a good citizen.
		}
		req.Endpoint = obj
		_, err := c.CreateEndpoint(ctx, req)
		if err != nil {
			if yterrors.ContainsErrorCode(err, ormerrors.CodeDuplicateObjectID) {
				fmt.Printf("Endpoint %s already exists\n", *obj.Meta.Id)
			} else {
				// TODO: errors?
				return fmt.Errorf("failed to create endpoint '%s': %w", *obj.Meta.Id, err)
			}
		} else {
			v, err := yson.MarshalFormat(obj, yson.FormatPretty)
			if err != nil {
				fmt.Printf("%s", fmt.Errorf("failed to marshal yson: %w", err))
			} else {
				fmt.Printf("Created %s\n", v)
			}
		}

	}
	// TODO: find non-equal - update (or delete and create)
	// TODO: find extra - delete
	// TODO: find missing - create
	return nil
}

func (y *YPSync) Expose(ctx context.Context, sets map[string]*Set) error {
	for zone, set := range sets {
		fmt.Printf("Syncing %s to YP(%s)...\n", set.Name.String(), zone)
		id := y.genSetID(set.Name)
		c, err := y.client.For(zone)
		if err != nil {
			return err
		}
		if err := y.createEndpointSet(ctx, c, id); err != nil {
			return err
		}
		if err := y.createEndpoints(ctx, c, set); err != nil {
			return err
		}
	}
	return nil
}
