package main

import (
	"fmt"
	"github.com/pulumi/pulumi-azure/sdk/v4/go/azure/compute"
	"github.com/pulumi/pulumi-azure/sdk/v4/go/azure/core"
	"github.com/pulumi/pulumi-azure/sdk/v4/go/azure/network"
	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
)

const (
	// Resource names
	resourceGroupName            = "example-resource-group"
	virtualNetworkName           = "example-virtual-network"
	subnetName                   = "example-subnet"
	networkInterfaceNameTemplate = "%s-network-interface"
	ipConfigurationNameTemplate  = "%s-ip-configuration"
	vmNameTemplate               = "az%s-%04d.rtc-oc.yandex.net"
	diskNameTemplate             = "%s-disk-%d"

	// Common
	eastUSRegion = "East US"

	// Network configuration
	vpcCIDR    = "10.0.0.0/16"
	subnetCIDR = "10.0.2.0/24"

	// VM configuration
	vmSize         = "Standard_DS1_v2"
	imagePublisher = "Canonical"
	imageOffer     = "UbuntuServer"
	imageSku       = "16.04-LTS"
	imageVersion   = "latest"
)

var (
	regionsShort = map[string]string{
		"East US": "eastus",
	}
)

func main() {
	pulumi.Run(func(ctx *pulumi.Context) error {
		rg, err := core.NewResourceGroup(ctx, resourceGroupName, &core.ResourceGroupArgs{
			Location: pulumi.String(eastUSRegion),
		})
		if err != nil {
			return err
		}
		sub, err := subnet(ctx, rg)
		if err != nil {
			return err
		}
		for i := 0; i < 10; i++ {
			vmName := fmt.Sprintf(vmNameTemplate, regionsShort[eastUSRegion], i)
			if _, err := virtualMachine(ctx, rg, sub, vmName); err != nil {
				return err
			}
		}
		return nil
	})
}

func subnet(ctx *pulumi.Context, rg *core.ResourceGroup) (*network.Subnet, error) {
	mainVirtualNetwork, err := network.NewVirtualNetwork(ctx, virtualNetworkName, &network.VirtualNetworkArgs{
		AddressSpaces:     pulumi.StringArray{pulumi.String(vpcCIDR)},
		Location:          rg.Location,
		ResourceGroupName: rg.Name,
	})
	if err != nil {
		return nil, err
	}
	return network.NewSubnet(ctx, subnetName, &network.SubnetArgs{
		ResourceGroupName:  rg.Name,
		VirtualNetworkName: mainVirtualNetwork.Name,
		AddressPrefixes:    pulumi.StringArray{pulumi.String(subnetCIDR)},
	})
}

func networkInterfaceIpConfiguration(subnet *network.Subnet, vmName string) *network.NetworkInterfaceIpConfigurationArgs {
	return &network.NetworkInterfaceIpConfigurationArgs{
		Name:                       pulumi.String(fmt.Sprintf(ipConfigurationNameTemplate, vmName)),
		SubnetId:                   subnet.ID(),
		PrivateIpAddressAllocation: pulumi.String("Dynamic"),
	}
}

func networkInterface(ctx *pulumi.Context, rg *core.ResourceGroup, sub *network.Subnet, vmName string) (*network.NetworkInterface, error) {
	return network.NewNetworkInterface(ctx, fmt.Sprintf(networkInterfaceNameTemplate, vmName), &network.NetworkInterfaceArgs{
		Location:          rg.Location,
		ResourceGroupName: rg.Name,
		IpConfigurations: network.NetworkInterfaceIpConfigurationArray{
			networkInterfaceIpConfiguration(sub, vmName),
		},
	})
}

func storageOsDisk(vmName string) *compute.VirtualMachineStorageOsDiskArgs {
	return &compute.VirtualMachineStorageOsDiskArgs{
		Name:            pulumi.String(fmt.Sprintf(diskNameTemplate, vmName, 1)),
		Caching:         pulumi.String("ReadWrite"),
		CreateOption:    pulumi.String("FromImage"),
		ManagedDiskType: pulumi.String("Standard_LRS"),
	}
}

func osProfile(vmName string) *compute.VirtualMachineOsProfileArgs {
	return &compute.VirtualMachineOsProfileArgs{
		ComputerName:  pulumi.String(vmName),
		AdminUsername: pulumi.String("testadmin"),
		AdminPassword: pulumi.String("Password1234!"),
	}
}

func osProfileLinuxConfig() *compute.VirtualMachineOsProfileLinuxConfigArgs {
	return &compute.VirtualMachineOsProfileLinuxConfigArgs{
		DisablePasswordAuthentication: pulumi.Bool(false),
	}
}

func virtualMachine(ctx *pulumi.Context, rg *core.ResourceGroup, sub *network.Subnet, vmName string) (*compute.VirtualMachine, error) {
	nic, err := networkInterface(ctx, rg, sub, vmName)
	if err != nil {
		return nil, err
	}
	return compute.NewVirtualMachine(ctx, vmName, &compute.VirtualMachineArgs{
		Location:          rg.Location,
		ResourceGroupName: rg.Name,
		NetworkInterfaceIds: pulumi.StringArray{
			nic.ID(),
		},
		VmSize: pulumi.String(vmSize),
		StorageImageReference: &compute.VirtualMachineStorageImageReferenceArgs{
			Publisher: pulumi.String(imagePublisher),
			Offer:     pulumi.String(imageOffer),
			Sku:       pulumi.String(imageSku),
			Version:   pulumi.String(imageVersion),
		},
		StorageOsDisk:        storageOsDisk(vmName),
		OsProfile:            osProfile(vmName),
		OsProfileLinuxConfig: osProfileLinuxConfig(),
		Tags:                 pulumi.StringMap{"environment": pulumi.String("staging")},
	})
}
