package main

import (
	"fmt"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/s3"
	"github.com/aws/aws-sdk-go/service/s3/s3manager"
	"io"
	"io/ioutil"
	"log"
	"os"
	"path"
)

type datastore struct {
	s         *session.Session
	bucket    string
	uploadDir string

	currentKeys []string
}

func (d *datastore) Keys() []string {
	return d.currentKeys
}

func (d *datastore) Object(name string) ([]byte, error) {
	fmt.Println("Reading object from local fs...")
	f, err := os.Open(path.Join(d.uploadDir, name))
	if err != nil {
		return nil, err
	}
	defer func() { _ = f.Close() }()
	content, err := io.ReadAll(f)
	if err != nil {
		return nil, err
	}
	return content, nil
}

func (d *datastore) Sync() error {
	fmt.Println("Syncing s3 object to local fs...")
	c := s3.New(d.s)
	keys := make([]string, 0)
	if err := c.ListObjectsPages(&s3.ListObjectsInput{
		Bucket: &d.bucket,
	}, func(p *s3.ListObjectsOutput, last bool) (shouldContinue bool) {
		for _, obj := range p.Contents {
			fmt.Println("Listed:", *obj.Key)
			keys = append(keys, *obj.Key)
		}
		return true
	}); err != nil {
		return err
	}

	for _, key := range keys {
		if err := d.downloadObject(key); err != nil {
			return err
		}
	}
	d.currentKeys = keys
	return nil
}

func (d *datastore) downloadObject(key string) error {
	downloader := s3manager.NewDownloader(d.s)

	cwd, err := os.Getwd()
	if err != nil {
		return err
	}

	temp, err := ioutil.TempFile(cwd, "getObjWithProgress-tmp-")
	if err != nil {
		return err
	}
	tempfileName := temp.Name()

	params := &s3.GetObjectInput{
		Bucket: aws.String(d.bucket),
		Key:    aws.String(key),
	}

	if _, err := downloader.Download(temp, params); err != nil {
		log.Printf("Download failed! Deleting tempfile: %s", tempfileName)
		_ = os.Remove(tempfileName)
		return err
	}

	if err := temp.Close(); err != nil {
		return err
	}

	if err := os.Rename(temp.Name(), path.Join(d.uploadDir, parseFilename(key))); err != nil {
		return err
	}

	log.Println("File downloaded! Available at:", path.Join(d.uploadDir, parseFilename(key)))
	return nil
}

func parseFilename(keyString string) (filename string) {
	_, file := path.Split(keyString)
	return file
}
