import time
import math
from collections import defaultdict

import pymongo


class Lacmus2HistoricalStorage(object):
    def __init__(self, mongo_client, config):
        self.db = mongo_client[config['HISTORICAL_DB_NAME']]
        self.config = config
        if not self.db.historical.options().get('capped'):
            self.db.create_collection('historical', capped=True,
                                      size=self.config['HISTORICAL_SIZE'])
            self.db.historical.create_indexes([
                pymongo.IndexModel([('chart', pymongo.ASCENDING),
                                    ('time', pymongo.DESCENDING)]),
            ])

    def _get_ts(self):
        return time.time()

    def write_point(self, chart_key, values):
        point = {'chart': chart_key,
                 'time': self._get_ts(),
                 'values': [(key, values[key]) for key in sorted(values)]}
        self.db.historical.insert_one(point)

    def get_graph(self, chart_key, from_time=None):
        fltr = {'chart': chart_key}
        if from_time is not None:
            fltr['time'] = {'$gt': from_time}
        sort = [('chart', pymongo.ASCENDING),
                ('time', pymongo.DESCENDING)]
        prj = {'_id': False}
        limit = self.config['HISTORICAL_MAX_POINTS']
        data = self.db.historical.find(fltr, prj).sort(sort)[:limit]
        all_keys = set()
        points = defaultdict(list)
        last_time = 0
        for point in data:
            last_time = max(last_time, point['time'])
            for key, value in point['values']:
                all_keys.add(key)
                points[key].append((point['time'], value))
        if not points:
            return [], None
        series = [
            {'name': key,
             'data': [{'x': int(point_time), 'y': point_value}
                      for point_time, point_value in reversed(points[key])]}
            for key in sorted(all_keys)
        ]
        return series, int(math.ceil(last_time))
