import re


NUMBERED_HOSTNAME = re.compile(
    r'(?P<prefix>.+?)(?P<number>\d+)(?P<suffix>[^\d]+$)'
)


def compactify_hostlist(hostlist):
    """
    Take an iterable of hostnames and makes it shell-compatibly compact,
    replacing number patterns like 'commonprefix-0123.commonsuffix'
    with strings like 'commonprefix-{0123..0128}.commonsuffix'.

    NB: Only works right if ``hostlist`` is sorted!

    >>> chl = compactify_hostlist
    >>> chl(['sas1-123.ya.ru', 'sas1-125.ya.ru', 'sas1-128.ya.ru'])
    ['sas1-{123,125,128}.ya.ru']
    >>> chl(['sas1-123.ya.ru', 'sas1-124.ya.ru'])
    ['sas1-{123,124}.ya.ru']
    >>> chl(['sas1-123.ya.ru', 'sas1-124.ya.ru', 'sas1-125.ya.ru'])
    ['sas1-{123..125}.ya.ru']
    >>> chl(['sas1-0001.ya.ru', 'sas1-0002.ya.ru', 'sas1-0003.ya.ru'])
    ['sas1-{0001..0003}.ya.ru']
    >>> chl(['sas1-0001.ya.ru', 'sas1-02.ya.ru', 'sas1-03.ya.ru'])
    ['sas1-{0001,02,03}.ya.ru']
    >>> chl(['sas1-0001.ya.ru', 'sas1-02a.ya.ru', 'sas1-03.ya.ru'])
    ['sas1-0001.ya.ru', 'sas1-02a.ya.ru', 'sas1-03.ya.ru']
    >>> chl(['sas123.ya.ru', 'sas124.ya.ru', 'sas125.ya.ru'])
    ['sas{123..125}.ya.ru']
    >>> chl(['sas1-1.ya.ru', 'sas1-2.ya.ru', 'sas2-3.ya.ru'])
    ['sas1-{1,2}.ya.ru', 'sas2-3.ya.ru']
    >>> chl(['a1.ya.ru', 'a2.ya.ru', 'a3.ya.ru', 'a7.ya.ru', 'a8.ya.ru'])
    ['a{{1..3},7,8}.ya.ru']
    >>> chl(['distbuild02dev.net', 'distbuild03dev.net'])
    ['distbuild{02,03}dev.net']
    >>> chl(['redb01e-dbs.kp.ya.net', 'redb02e-dbs.kp.ya.net',
    ...      'redb03e-dbs.kp.ya.net'])
    ['redb{01..03}e-dbs.kp.ya.net']
    """
    result = []
    cmpct = None

    def finalize_cmpct():
        nonlocal cmpct, result
        if cmpct is None:
            return
        prefix, suffix, ranges = cmpct
        if len(ranges) == 1:
            [(dleft, dright, sleft, sright)] = ranges
            if dleft == dright:
                result.append('%s%s%s' % (prefix, sleft, suffix))
            elif dleft + 1 == dright:
                result.append('%s{%s,%s}%s' % (prefix, sleft, sright, suffix))
            else:
                result.append('%s{%s..%s}%s' % (prefix, sleft, sright, suffix))
        else:
            middle = ','.join(
                sleft if dleft == dright else
                '%s,%s' % (sleft, sright) if dleft + 1 == dright else
                '{%s..%s}' % (sleft, sright)
                for dleft, dright, sleft, sright in ranges
            )
            result.append('%s{%s}%s' % (prefix, middle, suffix))
        cmpct = None

    for host in hostlist:
        match = NUMBERED_HOSTNAME.match(host)
        if match is None:
            finalize_cmpct()
            result.append(host)
            continue

        gd = match.groupdict()
        number = int(gd['number'])

        if cmpct:
            prefix, suffix, ranges = cmpct
            dleft, dright, sleft, sright = ranges[-1]
            if (gd['prefix'], gd['suffix']) == (prefix, suffix):
                if dright + 1 == number and len(sleft) == len(gd['number']):
                    ranges[-1][1] = number
                    ranges[-1][3] = gd['number']
                else:
                    ranges.append([number, number, gd['number'], gd['number']])
                continue
            finalize_cmpct()

        if cmpct is None:
            cmpct = (gd['prefix'], gd['suffix'],
                     [[number, number, gd['number'], gd['number']]])

    finalize_cmpct()

    return result
