import socket
import os
import re
import logging
import threading
import time

import flask
from flask.sessions import NullSession
import json
import msgpack
from pymongo import MongoClient

import lacmus2.redis
from lacmus2.historical import Lacmus2HistoricalStorage
from lacmus2.web.model import Lacmus2MongoStorage


def make_app():
    app = flask.Flask('lacmus2.api')
    # we do not need sessions in lacmus2 web api
    app.session_interface.session_class = NullSession

    app.config.from_object('lacmus2.web.config')
    app.config.from_envvar('LACMUS2_API_CONFIG')

    mongoclient = MongoClient(
        app.config['MONGODB_URI'],
        serverSelectionTimeoutMS=app.config['MONGODB_SERVER_SELECTION_TIMEOUT']
    )
    app.mstorage = Lacmus2MongoStorage(mongoclient,
                                       app.config['MONGODB_DB_NAME'],
                                       app.config)

    app.hstorage = Lacmus2HistoricalStorage(mongoclient, app.config)

    app.rstorage = lacmus2.redis.Lacmus2RedisStorage(app.config['REDIS'])

    app.automissing = AutoMissing(
        app.rstorage, max_age=app.config['EXPIRE_HOSTREPORTS_AFTER'],
    )
    app.automissing.start()

    app.add_url_rule('/v1/hostreport',
                     view_func=hostreport, methods=['POST'])
    app.add_url_rule('/v1/chart/<key>',
                     view_func=chart, methods=['GET'])
    app.add_url_rule('/v1/historical/<key>',
                     view_func=historical_data, methods=['GET'])
    app.add_url_rule('/v1/hosts/<key>',
                     view_func=list_hosts, methods=['GET'])

    node_info = {'hostname': socket.gethostname(),
                 'pid': os.getpid(),
                 'address': app.config['WSGI_LISTEN_ADDRESS']}
    node_info_str = json.dumps(node_info, sort_keys=True)

    app.logger.info('running on %(address)r host %(hostname)r, pid %(pid)d',
                    node_info)

    @app.errorhandler(Exception)
    def server_error(error):
        app.logger.exception(error)
        text = "Internal server error: %s\nNode info: %s" % (error,
                                                             node_info_str)
        return text, 500

    def stop_aside_threads():
        app.automissing.stop()

    app.cleanup = stop_aside_threads

    return app


class AutoMissing(threading.Thread):
    CLEANUP_EVERY = 10
    CLEANUP_BY = 1000

    def __init__(self, rstorage, max_age):
        self.rstorage = rstorage
        self.max_age = max_age
        self.logger = logging.getLogger('automissing')
        self._stopped = threading.Event()
        self._stopped.clear()
        super(AutoMissing, self).__init__()

    def run(self):
        while True:
            self._stopped.wait(timeout=self.CLEANUP_EVERY)
            if self._stopped.is_set():
                break
            try:
                n = self.rstorage.cleanup(self.max_age, limit=self.CLEANUP_BY)
                self.logger.info('cleaned up %d reports', n)
            except:
                self.logger.exception('')

    def stop(self):
        self._stopped.set()
        self.join()


def hostreport():
    reports = msgpack.loads(flask.request.data, encoding='utf8')
    for report in reports:
        hostname = report['host']
        timestamp = report['timestamp']
        for key, value in report['reports'].items():
            flask.current_app.rstorage.process_hostreport(
                hostname, timestamp, key, value
            )
    return flask.jsonify({'result': 'ok'})


def chart(key):
    result = flask.current_app.mstorage.get_volatile(
        vtype='chart', key_hash=key, full=True
    )
    flask.current_app.rstorage.mark_chart_as_viewed(key)
    if not flask.request.args.get('meta'):
        result = {'value': result['value'],
                  'time': int(time.time())}
    response = flask.jsonify(result)
    response.headers['Access-Control-Allow-Origin'] = (
        flask.current_app.config['ACCESS_CONTROL_ALLOW_ORIGIN']
    )
    return response


def list_hosts(key):
    value = flask.request.args.get('value') or None
    chart = flask.current_app.mstorage.get_volatile(
        vtype='chart', key_hash=key, full=True
    )
    page = flask.request.args.get('page', 0, type=int)
    hosts_on_page = flask.current_app.config['NUM_HOSTS_ON_PAGE']
    if flask.request.args.get('allhosts', False):
        hosts_on_page = flask.current_app.config['MAX_HOSTS_ON_PAGE']
    hosts, page, numpages = flask.current_app.rstorage.list_hosts(
        selector_vtype=chart['source']['selector_vtype'],
        selector_key=chart['source']['selector_key'],
        key=chart['source']['signal'],
        value=value,
        filters=chart['source']['filters'],
        page=page,
        hosts_on_page=hosts_on_page,
        compact=flask.request.args.get('compact', False),
    )
    hosts_on_page = flask.current_app.rstorage.NUM_HOSTS_ON_PAGE
    result = {'source': chart['source'], 'value': chart['value'],
              'page': page, 'numpages': numpages, 'hosts': hosts,
              'hosts_on_page': hosts_on_page}
    response = flask.jsonify(result)
    response.headers['Access-Control-Allow-Origin'] = (
        flask.current_app.config['ACCESS_CONTROL_ALLOW_ORIGIN']
    )
    return response


def historical_data(key):
    if not re.match('^[a-fA-F0-9]+$', key):
        flask.abort(400)
    from_time = flask.request.args.get('from_time')
    from_time = int(from_time) if from_time and from_time.isdigit() else None
    series, last_time = flask.current_app.hstorage.get_graph(key, from_time)
    series = series[::-1]
    if series and series[-1]['name'] == '':
        series = series[-1:] + series[:-1]
    response = flask.jsonify({'series': series, 'last_time': last_time})
    response.headers['Access-Control-Allow-Origin'] = (
        flask.current_app.config['ACCESS_CONTROL_ALLOW_ORIGIN']
    )
    return response
