import wtforms
from wtforms import validators
from wtforms.utils import unset_value
from wtforms.ext.csrf.form import SecureForm
from werkzeug.datastructures import MultiDict
from flask_wtf.form import Form
from flask.ext.babel import lazy_gettext as _


class _NoCSRFForm(Form):
    def __init__(self, *args, **kwargs):
        kwargs['csrf_enabled'] = False
        super(_NoCSRFForm, self).__init__(*args, **kwargs)
        del self.csrf_token

    @property
    def data(self):
        return super(SecureForm, self).data


class FormListField(wtforms.FieldList):
    def __init__(self, form, label, add_new_label, remove_this_label,
                 **kwargs):
        self.add_new_label = add_new_label
        self.remove_this_label = remove_this_label
        self.__form = form
        super(FormListField, self).__init__(wtforms.FormField(form), label,
                                            **kwargs)

    def process(self, formdata, data=unset_value):
        self.entries = []
        if data is unset_value or not data:
            try:
                data = self.default()
            except TypeError:
                data = self.default

        self.object_data = data

        if formdata:
            data_lists = {}
            max_index = 0
            for field in formdata:
                if field.startswith(self.name):
                    data_lists[field] = formdata.getlist(field)
                    max_index = max(max_index, len(data_lists[field]))
            for datalist in data_lists.values():
                datalist.extend([''] * (max_index - len(datalist)))
            for i in range(max_index):
                obj_data = {k: v[i] for k, v in data_lists.items()}
                self._add_entry(MultiDict(obj_data))
        else:
            for obj_data in data:
                self._add_entry(formdata, obj_data)

        while len(self.entries) < self.min_entries:
            self._add_entry(formdata)

    def _add_entry(self, formdata=None, data=unset_value):
        assert not self.max_entries or len(self.entries) < self.max_entries, \
            'You cannot have more than max_entries entries in this FieldList'
        field = self.unbound_field.bind(
            form=None, name=self.name, prefix=self._prefix, id=self.name,
            _meta=self.meta, translations=self._translations
        )
        field.process(formdata, data)
        self.entries.append(field)
        return field

    def get_template(self):
        return self.__form(formdata=None, data=None, prefix=self.name + '-')


def make_dashboard_form(signals, dashboard):
    class FiltersForm(_NoCSRFForm):
        inline = True

        def __init__(self, formdata=None, obj=None, *args, **kwargs):
            if obj:
                kwargs['data'] = dict(obj)
            super(FiltersForm, self).__init__(formdata, obj, *args, **kwargs)

    for key in sorted(signals):
        choices = [('', '')] + [(v, v) for v in sorted(signals[key])]
        select = wtforms.SelectField(key, choices=choices,
                                     validators=[validators.Optional()])
        setattr(FiltersForm, key, select)

    class ChartForm(_NoCSRFForm):
        signal = wtforms.SelectField(
            _('Signal to plot'), [validators.InputRequired()],
            choices=[('', '')] + [(k, k) for k in signals]
        )
        desc = wtforms.TextAreaField(_('Chart description'))
        selector = wtforms.TextAreaField(_('Selector'))
        filters = wtforms.FormField(FiltersForm)

        @property
        def data(self):
            data = super(ChartForm, self).data
            data['filters'] = sorted(
                (key, value)
                for key, value in data['filters'].items()
                if value
            )
            return data

    class DashboardForm(Form):
        name = wtforms.StringField(_('Dashboard name'),
                                   [validators.required()])
        desc = wtforms.TextAreaField(_('Dashboard description'))
        charts = FormListField(ChartForm, _('Charts'), _('Add a chart'),
                               _('Remove this chart'), min_entries=1)

    form = DashboardForm(data=dashboard)
    return form
