import json
import re

from genisys.web import model as genisys_model


class Lacmus2MongoStorage(genisys_model.BaseGenisysMongoStorage):
    _GENISYS_SELECTOR_RE = re.compile(
        r'^genisys@(?P<path>\w+(\.\w+)*)/(?P<rule>.+)$'
    )

    def __init__(self, client, db_name, config):
        self.config = config
        super(Lacmus2MongoStorage, self).__init__(client, db_name)

    def save_dashboard(self, username, dashboard_name, description, charts):
        now = self._get_ts()
        chart_keys = []

        for chart in charts:
            skey = selector = chart['selector'] = chart['selector'] or None
            chart_key = json.dumps((chart['signal'], chart['selector'],
                                    chart['filters']), sort_keys=True)
            chart_keys.append(chart_key)
            skey = selector
            if selector is None:
                svtype = None
            else:
                m = self._GENISYS_SELECTOR_RE.match(selector)
                if m is not None:
                    svtype = 'genisys_selector'
                    selector = m.groupdict()
                else:
                    svtype = 'selector'
            chart['selector_source'] = selector
            if skey is None:
                chart['selector_key'] = None
            else:
                chart['selector_key'] = genisys_model.volatile_key_hash(skey)
            chart['selector_vtype'] = svtype

        dashboard = {
            "creator": username,
            "name": dashboard_name,
            "desc": description,
            "mtime": now,
            "charts": [
                dict(chart, key=genisys_model.volatile_key_hash(chart_keys[i]))
                for i, chart in enumerate(charts)
            ],
        }
        self.db.dashboard.replace_one(
            {"creator": username, "name": dashboard_name},
            dashboard,
            upsert=True
        )

        for i, key in enumerate(chart_keys):
            chart = charts[i]
            if chart['selector_vtype'] is not None:
                self._ensure_volatile(
                    vtype=chart['selector_vtype'], key=chart['selector'],
                    source=chart['selector_source'], meta={}, now=now
                )
            self._ensure_volatile(vtype='chart', key=key, source=chart,
                                  meta={}, now=now)

        dkey = (username, dashboard_name)
        self._ensure_volatile(vtype='dashboard', key=dkey,
                              source={'charts': dashboard['charts']},
                              meta={}, now=now)

    def delete_dashboard(self, username, dashboard_name):
        result = self.db.dashboard.delete_one({'creator': username,
                                               'name': dashboard_name})
        dkey = (username, dashboard_name)
        self.db.volatile.delete_one(
            {'vtype': 'dashboard',
             'key': genisys_model.volatile_key_hash(dkey)}
        )
        return result.deleted_count > 0

    def list_dashboards(self, username):
        res = self.db.dashboard.find({'creator': username}, {'_id': False})
        return list(res.sort([('name', 1)]))

    def get_dashboard(self, username, dashboard_name):
        dashboard = list(self.db.dashboard.find(
            {'creator': username, 'name': dashboard_name},
            {'_id': False},
            limit=1
        ))
        if not dashboard:
            return None
        [dashboard] = dashboard
        return dashboard
