# coding=utf-8

import random
import json
import requests
import pandas as pd


FIELD_DATE = 'fielddate'


class StatException(Exception):

    def __init__(self, message):
        self.__message = message

    def __str__(self):
        return repr(self.__message)


class StatReport(object):

    def __init__(self, report, server, login, password, stat_request_timeout=5):
        self.__user = {'StatRobotUser': login, 'StatRobotPassword': password}
        self.__url = server + "/_api/report/"
        self.__report = report
        self.__stat_request_timeout = stat_request_timeout

    @property
    def report_path(self):
        return self.__report

    def create(self, scale, title, config):
        params = {
            'json_config': json.dumps(
                {
                    'user_config': config,
                    'title': title,
                }
            ),
            'name': self.__report,
            'scale': scale,
            'verbose': 1,
            '_request_timeout': self.__stat_request_timeout
        }
        r = requests.post(self.__url + 'config', headers=self.__user, data=params)
        if r.status_code != 200:
            raise StatException(r.text)
        return r.json()

    def delete_scale(self, scale):
        params = {
            'name': self.__report,
            'scale': scale,
            'verbose': 1,
            '_request_timeout': self.__stat_request_timeout
        }
        r = requests.post(self.__url + 'delete', data=params, headers=self.__user)
        if r.status_code != 200:
            return StatException(r.text)
        return r.json()

    def delete_report(self):
        params = {
            'name': self.__report,
            'verbose': 1,
            '_request_timeout': self.__stat_request_timeout
        }
        r = requests.post(self.__url + 'delete_report', data=params, headers=self.__user)
        if r.status_code != 200:
            return StatException(r.text)
        return r.json()

    def get_config(self):
        params = {
            'name': self.__report,
            '_request_timeout': self.__stat_request_timeout
        }
        r = requests.get(self.__url + 'config', params=params, headers=self.__user)
        if r.status_code != 200:
            raise StatException(r.text)
        return r.json()

    def exists(self):
        try:
            self.get_config()
            return True
        except StatException:
            return False

    def write_data(self, scale, data):
        if isinstance(data, pd.DataFrame):
            data = data.to_dict('records')
        params = {
            'name': self.__report,
            'scale': scale,
            'data': json.dumps({'values': data}),
            '_request_timeout': self.__stat_request_timeout
        }
        r = requests.post(self.__url + 'data', data=params, headers=self.__user)
        if r.status_code != 200:
            return StatException(r.text)
        return int(r.json()['message'].split(':')[1])


def config_from_df(df, dimensions):
    dimensions = set(dimensions)

    def type_converter(field, pdt):
        if pdt.startswith('int') or pdt.startswith('float'):
            return 'number'
        if field == FIELD_DATE:
            return 'date'
        return 'string'

    columns = {column:type_converter(column, df.dtypes[column].name) for column in df.columns.values}
    if not dimensions.issubset(set(columns.keys())):
        raise StatException('invalid dimensions: ' + str(dimensions))
    if FIELD_DATE not in dimensions:
        raise StatException('fielddate is missed from dimensions')
    return {
        'dimensions': [{cname: ctype} for cname, ctype in columns.iteritems() if cname in dimensions],
        'measures': [{cname: ctype} for cname, ctype in columns.iteritems() if cname not in dimensions],
    }


def printable(data):
    return unicode(str(data), 'unicode-escape')


def main():
    users = ['user1', 'user2']
    resource = ['resource1', 'resource2']
    data = [{
        FIELD_DATE: '2016-10-11 12:{:02}:00'.format(i % 60),
        'user': random.choice(users),
        'resource': random.choice(resource),
        'value': i
    } for i in xrange(59)]
    report = StatReport('Adhoc/inikifor/test1', 'https://stat-beta.yandex-team.ru', 'robot_robot-trencher', '')
    #print printable(report.get_config())
    #print report.write_data('i', data)
    print config_from_df(pd.DataFrame(data), [FIELD_DATE, 'resource', 'user'])
    report2 = StatReport('Adhoc/inikifor/new', 'https://stat-beta.yandex-team.ru', 'robot_robot-trencher', '')
    print report2.create('d', 'Новый отчет', config_from_df(pd.DataFrame(data), [FIELD_DATE, 'resource', 'user']))
    print report2.delete_report()


if __name__ == "__main__":
    main()
