#pragma once

#include <library/cpp/threading/future/async.h>
#include <library/cpp/threading/future/future.h>

#include <util/datetime/base.h>
#include <library/cpp/deprecated/atomic/atomic.h>
#include <util/system/condvar.h>
#include <util/system/mutex.h>
#include <util/thread/pool.h>

namespace NInfra {
    class IBackgroundThread {
    public:
        virtual ~IBackgroundThread(){};
        virtual void Start() = 0;
        virtual void Stop() = 0;
        virtual bool IsRunning() const = 0;
    };


    class TBackgroundThread: public IBackgroundThread {
    public:
        using TCallback = std::function<void()>;

    public:
        TBackgroundThread(TCallback&& callback, TDuration sleepTime)
            : Callback_(std::move(callback))
            , SleepTime_(sleepTime)
            , Running_(0)
        {
        }

        ~TBackgroundThread() {
            Stop();
        }

        void Start() override {
            if (AtomicCas(&Running_, 1, 0)) {
                ThreadPool_.Start(1);
                Thread_ = NThreading::Async([this] { Loop(); }, ThreadPool_);
            }
        }

        void Stop() override {
            if (AtomicCas(&Running_, 0, 1)) {
                CondVar_.Signal();
                Thread_.Wait();
                ThreadPool_.Stop();
            }
        }

        bool IsRunning() const override {
            return AtomicGet(Running_);
        }

    private:
        void Loop() {
            while (IsRunning()) {
                Callback_();

                TGuard<TMutex> guard(Mutex_);
                CondVar_.WaitT(Mutex_, SleepTime_, [this] { return !IsRunning(); });
            }
        }

    private:
        TCallback Callback_;
        TDuration SleepTime_;

        TAtomic Running_;
        TMutex Mutex_;
        TCondVar CondVar_;

        TThreadPool ThreadPool_;
        NThreading::TFuture<void> Thread_;
    };

}
