#pragma once

#include "abc_response.h"
#include "abc_model.h"

#include <infra/libs/clients/http_executer/http_executer.h>

#include <library/cpp/cgiparam/cgiparam.h>

#include <util/string/join.h>

namespace NInfra::NClients {

using TPair = std::pair<TString, TString>;

template<typename TResponse>
class TAbcRequest {
public:
    TAbcRequest(NHttpExecuter::THttpExecuterPtr executer, const TString& path, const TCgiParameters& params = {});

    TAbcRequest<TResponse>& SetPage(i32 page) {
        Parameters_.InsertEscaped(TStringBuf("page"), ToString(page));
        return *this;
    }

    TAbcRequest<TResponse>& SetPageSize(i32 pageSize) {
        Parameters_.InsertEscaped(TStringBuf("page_size"), ToString(pageSize));
        return *this;
    }

    TAbcRequest<TResponse>& SetFields(TVector<TString>&& fields) {
        Parameters_.InsertEscaped(TStringBuf("fields"), JoinSeq(",", fields));
        return *this;
    }

    TAbcRequest<TResponse>& SetQuery(TVector<TPair>&& query) {
        for (auto const& [key, val] : query) {
            Parameters_.InsertEscaped(TStringBuf(key), val);
        }
        return *this;
    }

    TAbcRequest<TResponse> SetOrdering(const TString& ordering) {
        Parameters_.InsertEscaped(TStringBuf("ordering"), ordering);
        return *this;
    }

    TAbcRequest<TResponse>& SetGtId(TMaybe<i64> id) {
        if (id.Defined()) {
            Parameters_.InsertEscaped(TStringBuf("id__gt"), ToString(*id));
        }
        return *this;
    }

    TResponse Execute(TLogFramePtr frame) const {
        TStringStream output = Executer_->Execute(GetPathWithParams(), NHttpExecuter::EHttpMethod::GET, frame);
        return TResponse(NJson::ReadJsonTree(&output, true));
    }

    TString GetPathWithParams() const {
        return Path_ + Parameters_.Print();
    }

protected:
    TCgiParameters Parameters_;

private:
    const NHttpExecuter::THttpExecuterPtr Executer_;
    const TString Path_;
};

class TConsumerRequest : public TAbcRequest<TAbcResponse<TServiceResource>>{
public:
    TConsumerRequest(NHttpExecuter::THttpExecuterPtr executer, const TString& url = "");

    TConsumerRequest& SetService(i64 id);

    TConsumerRequest& SetResourceTypeId(i64 resourceTypeId);
};

class TServiceMembersRequest : public TAbcRequest<TAbcResponse<TServiceMember>>{
public:
    TServiceMembersRequest(NHttpExecuter::THttpExecuterPtr executer, const TString& url = "");

    TServiceMembersRequest& SetState(const TString& state);
};

class TUpdateServiceConsumersRequest : public TAbcRequest<TAbcResponse<TServiceResource>> {
public:
    TUpdateServiceConsumersRequest(NHttpExecuter::THttpExecuterPtr executer, i64 id);

    TUpdateServiceConsumersRequest& SetState(const TString& state);
};

class TRolesScopesRequest : public TAbcRequest<TAbcResponse<TRolesScope>> {
public:
    TRolesScopesRequest(NHttpExecuter::THttpExecuterPtr executer);
};

class TGetAbcServicesRequest : public TAbcRequest<TAbcResponse<TAbcService>> {
public:
    TGetAbcServicesRequest(NHttpExecuter::THttpExecuterPtr executer, const TString& url = "");
};

} // namespace NInfra::NClients
