#include "golovan_request.h"

#include <library/cpp/json/json_reader.h>

#include <util/string/builder.h>
#include <util/string/cast.h>
#include <util/generic/yexception.h>

namespace NInfra::NClients {

TGolovanSingleStatRequest::TGolovanSingleStatRequest(NHttpExecuter::THttpExecuterPtr executer)
    : Executer_(executer)
{
}

TGolovanSingleStatRequest&  TGolovanSingleStatRequest::SetTag(const TString& key, const TString& value) {
    Labels_.insert({key, value});
    return *this;
}

TGolovanSingleStatRequest&  TGolovanSingleStatRequest::SetSignal(const TString& signal) {
    Signal_ = signal;
    return *this;
}

TGolovanSingleStatRequest&  TGolovanSingleStatRequest::SetPeriod(EGolovanPeriod period) {
    Period_ = period;
    return *this;
}

TGolovanSingleStatRequest&  TGolovanSingleStatRequest::SetHost(const TString& host) {
    Host_ = host;
    return *this;
}

TString TGolovanSingleStatRequest::GetPathWithParams() const {
    return "/hist/data";
}

TString TGolovanSingleStatRequest::GetSignals() const {
    TStringBuilder signals;
    for (const auto& [key, value] : Labels_) {
        if (!signals.Empty()) {
            signals << ";";
        }
        signals << key << "=" << value;
    }

    if (Signal_) {
        signals << ":" << Signal_;
    }
    return signals;
}

TString TGolovanSingleStatRequest::GetBody() const {
    ui64 et = Now().Seconds();
    ui64 st = et - ui64(Period_);

    return TStringBuilder() << "{\"ctxList\": [{\"name\": \"hist\""
                            << ",\"period\":" << (ui64)Period_
                            << ",\"st\":" << st
                            << ",\"signals\": [\""
                            << GetSignals()
                            << "\"], \"host\": \"" << Host_ << "\""
                            << ",\"et\":" << et
                            << ",\"id\": \"signal\"}]}";
}

double TGolovanSingleStatRequest::Execute(TLogFramePtr frame) const {
    TStringStream output = Executer_->Execute(GetPathWithParams(), NHttpExecuter::EHttpMethod::POST, frame, {}, GetBody());
    NJson::TJsonValue response = NJson::ReadJsonTree(&output, true);
    auto signals = response["response"]["signal"]["content"]["values"][GetSignals()].GetArray();
    if (signals.size() && signals[0].IsDouble()) {
        return signals[0].GetDouble(); // Interested in the latest guaranteed aggregated point
    }

    throw yexception() << "Not found signals in golovan response";
}

} // namespace NInfra::NClients
