#include "http_executer.h"

#include <infra/libs/sensors/macros.h>
#include <infra/libs/sensors/sensor.h>

#include <util/generic/strbuf.h>
#include <util/string/split.h>
#include <util/system/backtrace.h>

namespace {

const NInfra::TSensorGroup HTTP_EXECUTER_SENSOR_GROUP("http_executer");

TStringBuilder& AppendDelimeter(TStringBuilder& builder, char delimeter = '.') {
    if (!builder.empty() && builder.back() != delimeter) {
        builder << delimeter;
    }
    return builder;
}

TStringBuilder FormatUrl(const TStringBuf& input) {
    static constexpr char REPLACING_DELIM = '_';
    TStringBuilder output;

    for (const char c : input) {
        if (c == '.' || c == '/') {
            AppendDelimeter(output, REPLACING_DELIM);
            continue;
        }
        output << c;
    }
    if (output.EndsWith(REPLACING_DELIM)) {
        output.pop_back();
    }
    return output;
}

std::pair<TString, TString> GetAuthHeaderKeyValue(const TString& authSchema) {
    TString key;
    TString value;
    StringSplitter(authSchema).Split(':').Take(2).CollectInto(&key, &value);
    return {key, value};
}

NInfra::NLogEvent::THttpExecuterRequest::EHttpMethod ConvertHttpMethod(NInfra::NHttpExecuter::EHttpMethod httpMethod) {
    switch (httpMethod) {
        case NInfra::NHttpExecuter::EHttpMethod::GET:
            return NInfra::NLogEvent::THttpExecuterRequest::GET;
        case NInfra::NHttpExecuter::EHttpMethod::POST:
            return NInfra::NLogEvent::THttpExecuterRequest::POST;
        case NInfra::NHttpExecuter::EHttpMethod::PUT:
            return NInfra::NLogEvent::THttpExecuterRequest::PUT;
        case NInfra::NHttpExecuter::EHttpMethod::DELETE:
            return NInfra::NLogEvent::THttpExecuterRequest::DELETE;
        default:
            throw yexception() << "EObjectType " << httpMethod << " not supported";
    }
}

} // namespace

namespace NInfra::NHttpExecuter {

TString THttpExecuter::BuildSensorName(
    const TString& path
    , NInfra::NHttpExecuter::EHttpMethod httpMethod
    , TKeepAliveHttpClient::THttpCode code
    , const TString& sensorPathCustomName
) const {
    TStringBuilder sensorName;

    TStringBuf patchedProtocol;
    TStringBuf patchedHost;
    if (TStringBuf(Host_).TrySplit(TStringBuf("://"), patchedProtocol, patchedHost)) {
        sensorName << patchedProtocol;
    } else {
        patchedHost = Host_;
    }

    AppendDelimeter(sensorName) << FormatUrl(patchedHost);
    AppendDelimeter(sensorName) << FormatUrl(ApiPrefix_);

    if (sensorPathCustomName) {
        AppendDelimeter(sensorName) << sensorPathCustomName;
    } else {
        AppendDelimeter(sensorName) << FormatUrl(TStringBuf(path).NextTok("?"));
    }

    auto httpMethodStr = TStringBuilder() << httpMethod;

    return AppendDelimeter(sensorName)
        << AppendDelimeter(httpMethodStr)
        << ToString(code)[0] << "xx";
}

TStringStream THttpExecuter::Execute(
    const TString& path
    , EHttpMethod httpMethod
    , TLogFramePtr frame
    , THeaders&& headers
    , TStringBuf body
    , const TString& sensorPathCustomName
) const {
    NLogEvent::THttpExecuterRequest requestLog(
        Host_ + ApiPrefix_ + path
        , TString{body}
        , ConvertHttpMethod(httpMethod)
        , 0
        , ReadOnlyMode_
    );
    requestLog.MutableHeaders()->insert(headers.begin(), headers.end());

    if (ReadOnlyMode_ && httpMethod != EHttpMethod::GET) {
        frame->LogEvent(
            ELogPriority::TLOG_DEBUG
            , requestLog
        );
        return TStringStream();
    }

    // It is important that token puts to the headers after they are inserted into the requestLog.
    if (AuthToken_) {
        const auto headerKeyValuePair = GetAuthHeaderKeyValue(AuthSchema_);
        headers.emplace(
            headerKeyValuePair.first
            , TStringBuilder() << headerKeyValuePair.second << " " << AuthToken_
        );
    }

    TKeepAliveHttpClient client(Host_, Port_, SocketTimeout_, ConnectTimeout_);

    TDuration sleepDuration = MinDelay_;
    for (ui32 tryIdx = 0; tryIdx <= RetriesCount_; ++tryIdx) {
        requestLog.SetTryId(tryIdx);

        TStringStream output;
        THttpHeaders outHeaders;
        TKeepAliveHttpClient::THttpCode httpCode;
        try {
            frame->LogEvent(
                ELogPriority::TLOG_DEBUG
                ,  requestLog
            );

            httpCode = client.DoRequest(
                ToString(httpMethod)
                , ApiPrefix_ + path
                , body
                , &output
                , headers
                , &outHeaders
            );
        } catch (...) {
            if (tryIdx == RetriesCount_) {
                TStringStream backtrace;
                FormatBackTrace(&backtrace);
                ythrow yexception() << CurrentExceptionMessage() << '\n' << backtrace.Str();
            } else {
                SleepWithBackOff(&sleepDuration);
                continue;
            }
        }

        NON_STATIC_INFRA_RATE_SENSOR_X(HTTP_EXECUTER_SENSOR_GROUP, BuildSensorName(path, httpMethod, httpCode, sensorPathCustomName), 1);

        if (httpCode >= 200 && httpCode < 300) {
            return output;
        } else if (tryIdx == RetriesCount_) {
            throw THttpRequestException(httpCode)
                << "Got " << httpCode << " at " << Host_ << ApiPrefix_ << path
                << ", Response headers: " << ToString(outHeaders) << ", Response data: " << output.Str();
        }

        SleepWithBackOff(&sleepDuration);
    }

    Y_UNREACHABLE();
}

TStringStream THttpLoggableExecuter::Execute(
    const TString& path
    , EHttpMethod httpMethod
    , TLogFramePtr frame
    , THeaders&& headers
    , TStringBuf body
    , const TString& /* sensorPathCustomName */
) const {
    with_lock(FileMutex_) {
        TFileOutput logFileOutput(LogFile_);
        logFileOutput << "Url=" << path << ";"
                        << "Method=" << ToString(httpMethod) << ";"
                        << "Headers={";
        for (auto const& [key, val] : headers) {
            logFileOutput << "{" << key << ";" << val << "};";
        }
        logFileOutput << "};"
                        << "Body={" << body << "};\n";
    }
    return BaseExecuter_->Execute(path, httpMethod, frame, std::move(headers), body);
}

} // namespace NInfra::NHttpExecuter
