#pragma once

#include <infra/libs/clients/http_executer/config/config.pb.h>

#include <library/cpp/http/simple/http_client.h>

#include <infra/libs/logger/log_frame.h>

#include <util/generic/ptr.h>
#include <util/stream/file.h>
#include <util/stream/str.h>
#include <util/string/cast.h>
#include <util/system/mutex.h>

namespace NInfra::NHttpExecuter {

enum class EHttpMethod {
    GET,
    POST,
    PUT,
    DELETE,
};

class IHttpExecuter;
using THttpExecuterPtr = TAtomicSharedPtr<IHttpExecuter>;

class IHttpExecuter {
public:
    using THeaders = TSimpleHttpClient::THeaders;

    virtual TStringStream Execute(
        const TString& path
        , EHttpMethod httpMethod
        , TLogFramePtr frame
        , THeaders&& headers = {}
        , TStringBuf body = {}
        , const TString& sensorPathCustomName = ""
    ) const = 0;

    virtual ~IHttpExecuter() = default;
};

class THttpExecuter : public IHttpExecuter {
public:
    THttpExecuter(const THttpExecuterConfig& config)
        : Host_(config.GetHost())
        , Port_(config.GetPort())
        , SocketTimeout_(FromString<TDuration>(config.GetSocketTimeout()))
        , ConnectTimeout_(FromString<TDuration>(config.GetConnectTimeout()))
        , ApiPrefix_(config.GetApiPrefix())
        , AuthToken_(config.GetAuthToken())
        , AuthSchema_(config.GetAuthSchema())
        , RetriesCount_(config.GetRetries().GetCount())
        , MinDelay_(FromString<TDuration>(config.GetRetries().GetMinDelay()))
        , MaxDelay_(FromString<TDuration>(config.GetRetries().GetMaxDelay()))
        , BackOffFactor_(config.GetRetries().GetBackOffFactor())
        , ReadOnlyMode_(config.GetReadOnlyMode())
    {}

    virtual TStringStream Execute(
        const TString& path
        , EHttpMethod httpMethod
        , TLogFramePtr frame
        , THeaders&& headers = {}
        , TStringBuf body = {}
        , const TString& sensorPathCustomName = ""
    ) const override final;

private:
    TString BuildSensorName(
        const TString& path
        , NInfra::NHttpExecuter::EHttpMethod httpMethod
        , TKeepAliveHttpClient::THttpCode code
        , const TString& sensorPathCustomName = ""
    ) const;

    void SleepWithBackOff(TDuration* duration) const {
        Sleep(*duration);
        *duration = Min(MaxDelay_, *duration * BackOffFactor_);
    }

private:
    const TString Host_;
    const ui32 Port_;
    const TDuration SocketTimeout_;
    const TDuration ConnectTimeout_;
    const TString ApiPrefix_;
    const TString AuthToken_;
    const TString AuthSchema_;
    const ui32 RetriesCount_;
    const TDuration MinDelay_;
    const TDuration MaxDelay_;
    const ui64 BackOffFactor_;
    const bool ReadOnlyMode_;
};

class THttpLoggableExecuter : public IHttpExecuter {
public:
    THttpLoggableExecuter(THttpExecuterPtr baseExecuter, const TString& logPath)
        : BaseExecuter_(baseExecuter)
        , LogFile_(logPath, CreateAlways | WrOnly)
    {}

    virtual TStringStream Execute(
        const TString& path
        , EHttpMethod httpMethod
        , TLogFramePtr frame
        , THeaders&& headers = {}
        , TStringBuf body = {}
        , const TString& sensorPathCustomName = ""
    ) const;

private:
    const THttpExecuterPtr BaseExecuter_;
    TMutex FileMutex_;
    const TFile LogFile_;
};

} // namespace NInfra::NHttpExecuter
