#include "racktables_client.h"

#include <library/cpp/json/json_reader.h>

#include <util/generic/hash.h>
#include <util/stream/file.h>
#include <util/stream/str.h>
#include <util/string/builder.h>
#include <util/string/split.h>
#include <util/system/env.h>

namespace NInfra::NRackTables {

THashMap<TString, TString> TRackTablesClientMock::GetIP4ToFqdnMap(TLogFramePtr /* frame */) const {
    return {};
}

void TRackTablesClientMock::RemoveIP4(const TString& /*ip4*/, TLogFramePtr /* frame */) {}

void TRackTablesClientMock::UpdateIP4Fqdn(const TString& /*ip4*/, const TString& /* fqdn*/, TLogFramePtr /* frame */) {}

TVector<TNetworkProject> TRackTablesClientMock::LoadVmIds(TLogFramePtr /* frame */) const {
    return {};
}

THashMap<TString, const TVector<i64>> TRackTablesClientMock::LoadMacroProjectsIds(TLogFramePtr /* frame */) const {
    return {};
}

TVector<TMacroOwners> TRackTablesClientMock::LoadMacroOwners(TLogFramePtr /* frame */) const {
    return {};
}

TVector<TFirewallSection> TRackTablesClientMock::LoadFirewallSections(TLogFramePtr /* frame */) const {
    return {};
}

TVector<TIP4AddressPoolPtr> TRackTablesClientMock::LoadIP4AddressPools(TLogFramePtr /* frame */) const {
    return {};
}

///////////////////////////////////////////////////////////////

TRackTablesClient::TRackTablesClient(const TRackTablesClientConfig& config, NHttpExecuter::THttpExecuterPtr executer)
    : ApiPaths_(config.GetRackTablesPath())
    , Executer_(std::move(executer))
{
}

THashMap<TString, TString> TRackTablesClient::GetIP4ToFqdnMap(TLogFramePtr frame) const {
    TStringStream output = Executer_->Execute(ApiPaths_.GetGetPath(), NHttpExecuter::EHttpMethod::GET, frame);

    NJson::TJsonValue parsedJson = NJson::ReadJsonTree(&output, true);
    const TJsonMapType& parsedJsonMap = parsedJson.GetMapSafe();

    THashMap<TString, TString> currentRackTablesState;
    for (const auto& it : parsedJsonMap) {
        TString ip4 = it.first;
        TString fqdn = it.second["fqdn"].GetString();
        currentRackTablesState[ip4] = fqdn;
    }
    return currentRackTablesState;
}

void TRackTablesClient::RemoveIP4(const TString& ip4, TLogFramePtr frame) {
    TStringBuilder body;
    body << "{\"addr\": \"" << ip4 << "\"}";
    Executer_->Execute(ApiPaths_.GetUpdatePath(), NHttpExecuter::EHttpMethod::DELETE, frame, {}, body);
}

void TRackTablesClient::UpdateIP4Fqdn(const TString& ip4, const TString& fqdn, TLogFramePtr frame) {
    TStringBuilder body;
    body << "{\"fqdn\": \"" << fqdn << "\", \"addr\": \"" << ip4 << "\"}";
    Executer_->Execute(ApiPaths_.GetUpdatePath(), NHttpExecuter::EHttpMethod::PUT, frame, {}, body);
}

TVector<TNetworkProject> TRackTablesClient::LoadVmIds(TLogFramePtr frame) const {
    TVector<TNetworkProject> result;
    TVector<TString> lines;
    TStringStream input = Executer_->Execute(ApiPaths_.GetVmProjectPath(), NHttpExecuter::EHttpMethod::GET, frame);
    Split(input.Str().c_str(), "\n", lines);
    for (TString line : lines) {
        TVector<TString> buf;
        Split(line, "\t", buf);
        Y_ENSURE(buf.size() == 3, "Line does not contain exactly three words: " << line);
        i64 projectId = IntFromString<i64, 16>(buf[1]);
        result.push_back(TNetworkProject(projectId, buf[0], buf[2]));
    }
    return result;
}

THashMap<TString, const TVector<i64>> TRackTablesClient::LoadMacroProjectsIds(TLogFramePtr frame) const {
    TStringStream input = Executer_->Execute(ApiPaths_.GetMacroProjectsIdsPath(), NHttpExecuter::EHttpMethod::GET, frame);
    NJson::TJsonValue jsonRsp = NJson::ReadJsonTree(&input, true);
    THashMap<TString, const TVector<i64>> macro2ProjectsIds;
    for (const auto& [k, v] : jsonRsp.GetMapSafe()) {
        const auto& projectsIdsStr = v.GetArraySafe();
        TVector<i64> projectsIdsNum(projectsIdsStr.size());
        for (size_t i = 0; i < projectsIdsStr.size(); ++i) {
            projectsIdsNum[i] = IntFromString<i64, 16>(projectsIdsStr[i].GetStringSafe());
        }

        macro2ProjectsIds.emplace(k, std::move(projectsIdsNum));
    }

    return macro2ProjectsIds;
}

TVector<TMacroOwners> TRackTablesClient::LoadMacroOwners(TLogFramePtr frame) const {
    TStringStream input = Executer_->Execute(ApiPaths_.GetMacroOwnerPath(), NHttpExecuter::EHttpMethod::GET, frame);
    NJson::TJsonValue jsonRsp = NJson::ReadJsonTree(&input, true);
    const THashMap<TString, NJson::TJsonValue>& macroMap = jsonRsp.GetMapSafe();
    TVector<TMacroOwners> result;
    result.reserve(macroMap.size());
    for (auto const& [macro, macroInfo] : macroMap) {
        result.emplace_back(TMacroOwners(macro, macroInfo["owners"]));
    }
    return result;
}

TVector<TFirewallSection> TRackTablesClient::LoadFirewallSections(TLogFramePtr frame) const {
    TStringStream input = Executer_->Execute(ApiPaths_.GetFirewallSectionPath(), NHttpExecuter::EHttpMethod::GET, frame);
    const NJson::TJsonValue jsonRsp = NJson::ReadJsonTree(&input, true);
    const TDeque<NJson::TJsonValue>& sectionsArray = jsonRsp.GetArraySafe();
    TVector<TFirewallSection> result;
    result.reserve(sectionsArray.size());
    for (const NJson::TJsonValue& json : sectionsArray ) {
        result.emplace_back(TFirewallSection(json));
    }
    return result;
}

TVector<TIP4AddressPoolPtr> TRackTablesClient::LoadIP4AddressPools(TLogFramePtr frame) const {
    TStringStream input = Executer_->Execute(ApiPaths_.GetIP4AddressPoolsPath(), NHttpExecuter::EHttpMethod::GET, frame);
    const NJson::TJsonValue jsonRsp = NJson::ReadJsonTree(&input, true);
    const TDeque<NJson::TJsonValue>& poolsArray = jsonRsp.GetMapSafe().at("pools").GetArraySafe();
    TVector<TIP4AddressPoolPtr> result;
    result.reserve(poolsArray.size());
    for (const NJson::TJsonValue& json : poolsArray) {
        result.emplace_back(MakeAtomicShared<TIP4AddressPool>(json));
    }
    return result;
}

///////////////////////////////////////////////////////////////

const TString TRackTablesClientLocal::LOG_FILEPATH = "racktables.log";

TRackTablesClientLocal::TRackTablesClientLocal()
    : LogFile_(LOG_FILEPATH, CreateAlways | WrOnly)
    , RackTablesState_({{"0.0.0.0", "dummy.fqdn"}})
{
}

THashMap<TString, TString> TRackTablesClientLocal::GetIP4ToFqdnMap(TLogFramePtr /* frame */) const {
    TGuard<TMutex> guard(LogMutex_);

    TFileOutput logFileOutput(LogFile_);
    logFileOutput << "Get mapping request\n";
    return RackTablesState_;
}

void TRackTablesClientLocal::RemoveIP4(const TString& ip4, TLogFramePtr /* frame */) {
    TGuard<TMutex> guard(LogMutex_);

    TFileOutput logFileOutput(LogFile_);
    logFileOutput << "Remove request = {\"ip4\":\"" << ip4 << "\"}\n";
    RackTablesState_.erase(ip4);
}

void TRackTablesClientLocal::UpdateIP4Fqdn(const TString& ip4, const TString& fqdn, TLogFramePtr /* frame */) {
    TGuard<TMutex> guard(LogMutex_);

    TFileOutput logFileOutput(LogFile_);
    logFileOutput << "Change request = {\"ip4\":\"" << ip4 << "\", \"fqdn\":\"" << fqdn << "\"}\n";
    RackTablesState_[ip4] = fqdn;
}

TVector<TNetworkProject> TRackTablesClientLocal::LoadVmIds(TLogFramePtr /* frame */) const {
    TGuard<TMutex> guard(LogMutex_);

    TFileOutput logFileOutput(LogFile_);
    logFileOutput << "LoadVmIds request\n";
    return {TNetworkProject(228, "_MACRO_", "SECTION")};
}

THashMap<TString, const TVector<i64>> TRackTablesClientLocal::LoadMacroProjectsIds(TLogFramePtr /* frame */) const {
    TGuard<TMutex> guard(LogMutex_);

    TFileOutput logFileOutput(LogFile_);
    logFileOutput << "LoadMacroProjectsIds request\n";

    return {{"_MACRO_", {228, 338}}};
}

TVector<TMacroOwners> TRackTablesClientLocal::LoadMacroOwners(TLogFramePtr /* frame */) const {
    TGuard<TMutex> guard(LogMutex_);
    TFileOutput logFileOutput(LogFile_);
    logFileOutput << "LoadMacroOwners request\n";

    NJson::TJsonValue owners;

    NJson::TJsonValue owner1;
    owner1.InsertValue("name", "test");
    owner1.InsertValue("type", "user");
    owners.AppendValue(owner1);

    NJson::TJsonValue owner2;
    owner2.InsertValue("name", "staff_department");
    owner2.InsertValue("type", "department");
    owners.AppendValue(owner2);

    NJson::TJsonValue owner3;
    owner3.InsertValue("name", "svc_service_slug");
    owner3.InsertValue("type", "service");
    owners.AppendValue(owner3);

    NJson::TJsonValue owner4;
    owner4.InsertValue("name", "svc_service_slug_scope_slug");
    owner4.InsertValue("type", "servicerole");
    owners.AppendValue(owner4);

    NJson::TJsonValue owner5;
    owner5.InsertValue("name", "svc_service_slug_fake_scope_slug_fake");
    owner5.InsertValue("type", "fake");
    owners.AppendValue(owner5);

    return {TMacroOwners("_MACRO_", owners)};
}

TVector<TFirewallSection> TRackTablesClientLocal::LoadFirewallSections(TLogFramePtr /* frame */) const {
    TGuard<TMutex> guard(LogMutex_);
    TFileOutput logFileOutput(LogFile_);
    logFileOutput << "LoadFirewallSections request\n";

    NJson::TJsonValue firewallSectionJson;
    firewallSectionJson.InsertValue("name", "name");

    NJson::TJsonValue owners;
    NJson::TJsonValue owner;
    owner.InsertValue("name", "test");
    owner.InsertValue("type", "user");
    owners.AppendValue(owner);
    firewallSectionJson.InsertValue("owners", owners);

    NJson::TJsonValue flags;
    flags.AppendValue("VS");
    firewallSectionJson.InsertValue("flags", flags);

    NJson::TJsonValue scope;
    scope.AppendValue("10d2ad9@2a02:6b8:0:3400:0:71d:0:11a/40");
    scope.AppendValue("2a02:6b8:0:3400::3:116/42");
    scope.AppendValue("141.8.146.116");
    firewallSectionJson.InsertValue("scope", scope);

    NJson::TJsonValue firewallSectionWithoutOwnersAndScopeJson;
    firewallSectionWithoutOwnersAndScopeJson.InsertValue("name", "emptiness");
    firewallSectionWithoutOwnersAndScopeJson.InsertValue("flags", flags);
    return {TFirewallSection(firewallSectionJson)
        , TFirewallSection(firewallSectionWithoutOwnersAndScopeJson)};
}

TVector<TIP4AddressPoolPtr> TRackTablesClientLocal::LoadIP4AddressPools(TLogFramePtr /* frame */) const {
    TGuard<TMutex> guard(LogMutex_);
    TFileOutput logFileOutput(LogFile_);
    logFileOutput << "LoadIP4AddressPools request\n";

    static const TString location = "IVA";

    // Values (1) and (2) are associated with hardcoded values
    // from yp/inet_mngr/tests/test.py.

    NJson::TJsonValue pool;
    pool["stable_id"] = "zora-pool";
    pool["stable_name"] = "Зора*пул_228";
    pool["owners"].AppendValue("staff:228");
    pool["owners"].AppendValue("zora-robot");
    pool["cidr"].AppendValue({});
    pool["cidr"][0]["subnet"] = "13.14.15.16/28";  // (1)
    pool["cidr"][0]["location"] = location;
    pool["cidr"].AppendValue({});
    pool["cidr"][1]["subnet"] = "33.22.11.0/30";  // (2)
    pool["cidr"][1]["location"] = location;

    return {
        MakeAtomicShared<TIP4AddressPool>(pool),
    };
}

///////////////////////////////////////////////////////////////

} // namespace NInfra::NRackTables

