#include "racktables_model.h"

#include <kernel/translit/translit.h>

#include <util/digest/murmur.h>
#include <util/network/ip.h>
#include <util/string/builder.h>
#include <util/string/cast.h>
#include <util/string/join.h>
#include <util/string/split.h>

#include <cctype>

#include <netinet/in.h>

namespace NInfra::NRackTables {

TNetworkProject::TNetworkProject(i64 projectId, TString macro, TString section)
    : ProjectId_(projectId)
    , Macro_(macro)
    , Section_(section)
{
}

void TNetworkProject::SetOwners(const TVector<TString>& owners) {
    Owners_ = owners;
}

void TNetworkProject::SetProjectId(const i64 projectId) {
    ProjectId_ = projectId;
}

i64 TNetworkProject::GetProjectId() const {
    return ProjectId_;
}

TString TNetworkProject::GetMacro() const {
    return Macro_;
}

TString TNetworkProject::GetSection() const {
    return Section_;
}

const TVector<TString>& TNetworkProject::GetOwners() const {
    return Owners_;
}

ui64 TNetworkProject::GetHash() const {
    TStringBuilder buf;
    buf << ToString(ProjectId_) << "#";
    buf << Macro_ << "#" << Section_ << "#";
    buf << JoinSeq("@", Owners_);
    return MurmurHash<ui64>(buf.data(), buf.size());
}

///////////////////////////////////////////////////////////////

TOwner::TOwner(const NJson::TJsonValue& json)
    : Type_(json["type"].GetString())
    , Name_(json["name"].GetString())
{
}

TString TOwner::GetType() const {
    return Type_;
}

TString TOwner::GetName() const {
    return Name_;
}

///////////////////////////////////////////////////////////////

TMacroOwners::TMacroOwners(TString macro, const NJson::TJsonValue& json)
    : Macro_(macro)
    , Owners_(json.GetArray().begin(), json.GetArray().end())
{
}

TString TMacroOwners::GetMacro() const {
    return Macro_;
}

const TVector<TOwner>& TMacroOwners::GetOwners() const {
    return Owners_;
}

///////////////////////////////////////////////////////////////

TFirewallSection::TFirewallSection(const NJson::TJsonValue& json)
    : Name_(json["name"].GetString())
    , Owners_(json["owners"].GetArray().begin(), json["owners"].GetArray().end())
{
    for (const NJson::TJsonValue& flagJson : json["flags"].GetArray()) {
        Flags_.emplace_back(flagJson.GetString());
    }

    for (const NJson::TJsonValue& scopeJson : json["scope"].GetArray()) {
        TVector<TString> scopeParts;
        Split(scopeJson.GetString(), "@", scopeParts);
        if (scopeParts[0].Contains("/")) {
            continue;
        }

        TString scope = scopeParts[0];
        if (scopeParts.size() == 2) {
            scope = scopeParts[1];
        }
        TVector<TString> addressAndMask;
        if (scope.Contains("/")) {
            Split(scope, "/", addressAndMask);
            Scope_.emplace_back(Ip4Or6FromString(addressAndMask[0].c_str())); //address
        } else {
            Scope_.emplace_back(Ip4Or6FromString(scope.c_str()));
        }
    }
}

TString TFirewallSection::GetName() const {
    return Name_;
}

const TVector<TOwner>& TFirewallSection::GetOwners() const {
    return Owners_;
}

const TVector<TString>& TFirewallSection::GetFlags() const {
    return Flags_;
}

const TVector<TIp4Or6>& TFirewallSection::GetScope() const {
    return Scope_;
}

ui64 TFirewallSection::GetHash() const {
    TStringBuilder buf;
    buf << Name_ << "#";
    for (TOwner owner : Owners_) {
        buf << owner.GetName() << "#" << owner.GetType() << "@";
    }

    buf << JoinSeq("@", Flags_);
    for (const TIp4Or6& address : Scope_) {
        buf << "#" << Ip4Or6ToString(address);
    }
    return MurmurHash<ui64>(buf.data(), buf.size());
}

///////////////////////////////////////////////////////////////

TInternetAddress::TInternetAddress(const TString& address)
    : TInternetAddress(ntohl(IpFromString(address.c_str())))
{
}

TInternetAddress::TInternetAddress(ui32 value)
    : IP4AddresValue_(value)
{
}

bool TInternetAddress::operator==(const TInternetAddress& rhs) const {
    return IP4AddresValue_ == rhs.IP4AddresValue_;
}

TIpHost TInternetAddress::GetRawValue() const {
    return htonl(IP4AddresValue_);
}

ui32 TInternetAddress::GetValue() const {
    return IP4AddresValue_;
}

TString TInternetAddress::ToString() const {
    return IpToString(GetRawValue());
}

void TInternetAddress::Advance(ui32 delta) {
    IP4AddresValue_ += delta;
}

///////////////////////////////////////////////////////////////

TInternetAddressIterator::TInternetAddressIterator(TInternetAddress address, ui32 offset)
    : InternetAddress_(address)
{
    InternetAddress_.Advance(offset);
}

bool TInternetAddressIterator::operator==(const TInternetAddressIterator& rhs) const {
    return InternetAddress_ == rhs.InternetAddress_;
}

bool TInternetAddressIterator::operator!=(const TInternetAddressIterator& rhs) const {
    return !(*this == rhs);
}

TInternetAddressIterator& TInternetAddressIterator::operator++() {
    InternetAddress_.Advance(1);
    return *this;
}

TInternetAddressIterator TInternetAddressIterator::operator++(int) {
    TInternetAddressIterator retval(*this);
    ++(*this);
    return retval;
}

TInternetAddress TInternetAddressIterator::operator*() const {
    return InternetAddress_;
}

const TInternetAddress* TInternetAddressIterator::operator->() const {
    return &InternetAddress_;
}

///////////////////////////////////////////////////////////////

TSubnet::TSubnet(TString cidrNotation)
    : CidrNotation_(std::move(cidrNotation))
{
    TStringBuf address;
    TStringBuf subnetPower;
    Y_ENSURE(TStringBuf(CidrNotation_).TrySplit('/', address, subnetPower));

    FirstInternetAddress_ = TInternetAddress(TString{address});

    const int power = FromString<int>(subnetPower);
    Y_ENSURE(0 <= power, "Invalid subnet mask: number must be positive");
    Y_ENSURE(power <= 32, "Invalid subnet mask");
    SubnetSize_ = 1ull << (32 - power);
}

TString TSubnet::GetCidrNotation() const {
    return CidrNotation_;
}

TInternetAddressIterator TSubnet::begin() const {
    return TInternetAddressIterator(FirstInternetAddress_);
}

TInternetAddressIterator TSubnet::end() const {
    return TInternetAddressIterator(FirstInternetAddress_, SubnetSize_);
}

bool TSubnet::Contains(const TInternetAddress address) const {
    const ui64 left = FirstInternetAddress_.GetValue();
    const ui64 right = left + SubnetSize_;
    const ui64 pos = address.GetValue();
    return (left <= pos) && (pos < right);
}

///////////////////////////////////////////////////////////////

TSubPool::TSubPool(TString location, TString cidrNotation)
    : Subnet_(std::move(cidrNotation))
    , Location_(std::move(location))
{
}

const TSubnet& TSubPool::GetSubnet() const {
    return Subnet_;
}

TString TSubPool::GetLocation() const {
    return Location_;
}

///////////////////////////////////////////////////////////////

TString TranslatePoolNameToId(TString name) {
    name = TransliterateBySymbol(name);
    name.to_lower();
    for (size_t pos = 0; pos < name.size(); ++pos) {
        if (!std::isalnum(name[pos])) {
            name[pos] = '-';
        }
    }
    return name;
}

TVector<TString> CastStrings(const NJson::TJsonValue& json) {
    TVector<TString> result;
    for (const auto& value : json.GetArray()) {
        result.push_back(value.GetStringSafe());
    }
    return result;
}

TVector<TSubPool> ParseAndFilterSubnets(const NJson::TJsonValue& subnetsJson) {
    TVector<TSubPool> result;
    for (const auto& subnetJson : subnetsJson.GetArraySafe()) {
        const auto& subnet = subnetJson.GetMapSafe();
        result.emplace_back(subnet.at("location").GetStringSafe(), subnet.at("subnet").GetStringSafe());
    }
    return result;
}

///////////////////////////////////////////////////////////////

TIP4AddressPool::TIP4AddressPool(const NJson::TJsonValue& json)
    : StableId_(json.GetMapSafe().at("stable_id").GetStringSafe())
    , StableName_(json.GetMapSafe().at("stable_name").GetStringSafe())
    , Owners_(CastStrings(json.GetMapSafe().at("owners")))
    , Subnets_(ParseAndFilterSubnets(json.GetMapSafe().at("cidr")))
{
}

TString TIP4AddressPool::GetStableId() const {
    return StableId_;
}

TString TIP4AddressPool::GetStableName() const {
    return StableName_;
}

const TVector<TString>& TIP4AddressPool::GetOwners() const {
    return Owners_;
}

const TVector<TSubPool>& TIP4AddressPool::GetSubnets() const {
    return Subnets_;
}

///////////////////////////////////////////////////////////////

} // namespace NInfra::NRackTables
