#pragma once

#include <library/cpp/json/json_reader.h>

#include <util/draft/ip.h>
#include <util/generic/strbuf.h>
#include <util/generic/string.h>
#include <util/generic/vector.h>

namespace NInfra::NRackTables {

class TNetworkProject {
public:
    TNetworkProject(i64 projectId, TString macro, TString section);

    void SetOwners(const TVector<TString>& owners);

    void SetProjectId(const i64);

    i64 GetProjectId() const;

    TString GetMacro() const;

    TString GetSection() const;

    const TVector<TString>& GetOwners() const;

    ui64 GetHash() const;

private:
    i64 ProjectId_;
    const TString Macro_;
    const TString Section_;
    TVector<TString> Owners_;
};

class TOwner {
public:
    TOwner(const NJson::TJsonValue& json);

    TString GetType() const;

    TString GetName() const;

private:
    const TString Type_;
    const TString Name_;
};

class TMacroOwners{
public:
    TMacroOwners(TString macro, const NJson::TJsonValue& json);

    TString GetMacro() const;

    const TVector<TOwner>& GetOwners() const;

private:
    const TString Macro_;
    const TVector<TOwner> Owners_;
};

class TFirewallSection {
public:
    TFirewallSection(const NJson::TJsonValue& json);

    TString GetName() const;

    const TVector<TOwner>& GetOwners() const;

    const TVector<TString>& GetFlags() const;

    const TVector<TIp4Or6>& GetScope() const;

    ui64 GetHash() const;

private:
    const TString Name_;
    const TVector<TOwner> Owners_;
    TVector<TString> Flags_;
    TVector<TIp4Or6> Scope_;
};

class TInternetAddress {
public:
    explicit TInternetAddress(const TString& address);
    explicit TInternetAddress(ui32 value);

    TInternetAddress() = default;

    bool operator==(const TInternetAddress& rhs) const;

    TIpHost GetRawValue() const;

    ui32 GetValue() const;

    TString ToString() const;

    void Advance(ui32 delta);

private:
    ui32 IP4AddresValue_ = 0;
};

class TInternetAddressIterator {
public:
    explicit TInternetAddressIterator(TInternetAddress address, ui32 offset = 0);
    TInternetAddressIterator(const TInternetAddressIterator&) = default;
    TInternetAddressIterator(TInternetAddressIterator&&) = default;

    bool operator==(const TInternetAddressIterator& rhs) const;

    bool operator!=(const TInternetAddressIterator& rhs) const;

    TInternetAddressIterator& operator++();

    TInternetAddressIterator operator++(int);

    TInternetAddress operator*() const;

    const TInternetAddress* operator->() const;

private:
    TInternetAddress InternetAddress_;
};

class TSubnet {
public:
    TSubnet(TString cidrNotation);

    TString GetCidrNotation() const;

    TInternetAddressIterator begin() const;
    TInternetAddressIterator end() const;

    bool Contains(const TInternetAddress address) const;

protected:
    TString CidrNotation_;
    TInternetAddress FirstInternetAddress_;
    ui64 SubnetSize_ = 0;
};

class TSubPool {
public:
    TSubPool(TString location, TString cidrNotation);

    const TSubnet& GetSubnet() const;
    TString GetLocation() const;

private:
    TSubnet Subnet_;
    TString Location_;
};

class TIP4AddressPool {
public:
    TIP4AddressPool(const NJson::TJsonValue& json);

    TString GetStableId() const;
    TString GetStableName() const;
    const TVector<TString>& GetOwners() const;
    const TVector<TSubPool>& GetSubnets() const;

private:
    const TString StableId_;
    const TString StableName_;
    const TVector<TString> Owners_;
    TVector<TSubPool> Subnets_;
};

using TIP4AddressPoolPtr = TAtomicSharedPtr<TIP4AddressPool>;

} // namespace NInfra::NRackTables
