#include "solomon_request.h"

#include <library/cpp/json/json_reader.h>

#include <util/string/builder.h>
#include <util/string/cast.h>
#include <util/generic/yexception.h>

namespace NInfra::NClients {

TSolomonSingleStatRequest::TSolomonSingleStatRequest(NHttpExecuter::THttpExecuterPtr executer)
    : Executer_(executer)
{
}

TSolomonSingleStatRequest& TSolomonSingleStatRequest::SetProject(const TString& project) {
    Project_ = project;
    return *this;
}

TSolomonSingleStatRequest& TSolomonSingleStatRequest::SetService(const TString& service) {
    Service_ = service;
    return *this;
}

TSolomonSingleStatRequest& TSolomonSingleStatRequest::SetCluster(const TString& cluster) {
    Cluster_ = cluster;
    return *this;
}

TSolomonSingleStatRequest& TSolomonSingleStatRequest::SetHost(const TString& host) {
    Host_ = host;
    return *this;
}

TSolomonSingleStatRequest& TSolomonSingleStatRequest::SetSensor(const TString& sensor) {
    Y_ENSURE(!Path_.Defined(), "Sensor and path fields are not supported at the same time");
    Sensor_ = sensor;
    return *this;
}

TSolomonSingleStatRequest& TSolomonSingleStatRequest::SetPath(const TString& path) {
    Y_ENSURE(!Sensor_.Defined(), "Sensor and path fields are not supported at the same time");
    Path_ = path;
    return *this;
}

TSolomonSingleStatRequest& TSolomonSingleStatRequest::SetAggregationType(const TString& aggregationType) {
    AggregationType_ = aggregationType;
    return *this;
}

TSolomonSingleStatRequest&TSolomonSingleStatRequest::SetPeriod(const TDuration& period) {
    Period_ = period;
    return *this;
}

TSolomonSingleStatRequest& TSolomonSingleStatRequest::AddAdvancedAttribute(const TString& key, const TString& value) {
    AdvancedAttributes_.insert({key, value});
    return *this;
}

TString TSolomonSingleStatRequest::GetPathWithParams() const {
    return TStringBuilder() << "/projects/" << Project_ << "/sensors/data";
}

TString TSolomonSingleStatRequest::GetProgram() const {
    TStringBuilder res;
    res << "\"program\":\"" << AggregationType_ << "({";
    if (Sensor_.Defined()) {
        res << "sensor='" << Sensor_.GetRef() << "',";
    } else {
        res << "path='" << Path_.GetRef() << "',";
    }

    for (const auto& [key, value] : AdvancedAttributes_) {
        res << key << "='"<< value << "',";
    }

    return res << "host='" << Host_ << "',"
               << "service='" << Service_ << "'})\"";
}

TString TSolomonSingleStatRequest::GetBody() const {
    TInstant et = Now();
    TInstant st = et - Period_;
    return TStringBuilder() << "{" << GetProgram() << ","
                            << "\"from\":\"" << st.ToString() << "\","
                            << "\"to\":\"" << et.ToString() << "\","
                            << "\"forceCluster\": \"" << Cluster_ << "\"}";
}

double TSolomonSingleStatRequest::Execute(TLogFramePtr frame) const {
    TStringStream output = Executer_->Execute(
        GetPathWithParams()
        , NHttpExecuter::EHttpMethod::POST
        , frame
        , {
            {"Content-Type", "application/json; charset=utf-8"}
        }
        , GetBody()
    );

    NJson::TJsonValue response = NJson::ReadJsonTree(&output, true);
    if (response["scalar"].IsDouble()) {
        return response["scalar"].GetDouble();
    }

    throw yexception() << "Not found signals in solomon response: \n" << output.Str();
}

} // namespace NInfra::NClients
